
unit autosp3d;

// ==========================================================================
//  autospace3d - version 0.90 / o3.o8.2ooo
//  freepascal (clax) keyframing & (opengl) 3d engine - main system code.
// --------------------------------------------------------------------------
//  copyright (c)2ooo. remage / fresh!mindworkz.
// --------------------------------------------------------------------------

{x$define as_debug}

// ==========================================================================

interface

uses
  as_type, as_vect, as_matr, as_quat, as_splin, as_cam;

const
  as3d_version : pchar = 'autospace3d - version 0.90 / o3.o8.2ooo';

var
  as3d_scene : as_pscene;
  as3d_camera : as_pcamera;
  as3d_flags : longint;

// --------------------------------------------------------------------------

function as3d_init( flags: longint ): longint;
function as3d_done: longint;

function as3d_geterror( code: longint ): string;

function as3d_getframes( var start, ends: single ): longint;
function as3d_getframe( var frame: single ): longint;
function as3d_setframe( frame: single ): longint;

function as3d_setactivescene( scene: as_pscene ): longint;
function as3d_getactivescene( var scene: as_pscene ): longint;

function as3d_setactivecamera( camera: as_pcamera ): longint;
function as3d_getactivecamera( var camera: as_pcamera ): longint;

function as3d_byname( var node: as_pwnode; name: string ): longint;
function as3d_byid( var node: as_pwnode; id: longint ): longint;
function as3d_findfirst( var node: as_pwnode; attr: longint ): longint;
function as3d_findnext( var node: as_pwnode; attr: longint ): longint;

function as3d_allocscene( var scene: as_pscene ): longint;

function as3d_addworld( typ: longint; obj: pointer ): longint;
function as3d_addtrack( typ, id, parent: longint; track, obj: pointer ): longint;
function as3d_settrack( typ, id: longint; track: as_ptrack ): longint;

function as3d_freeworld( scene: as_pscene ): longint;
function as3d_freemotion( scene: as_pscene ): longint;
function as3d_freescene( scene: as_pscene ): longint;

procedure as3d_calcnormals;

function as3d_getkeyfloat( var out: single; track: as_ptrack; frame: single ): longint;
function as3d_getkeycolor( var out: as_color; track: as_ptrack; frame: single ): longint;
function as3d_getkeyvect( var out: as_vector; track: as_ptrack; frame: single ): longint;
function as3d_getkeyquat( var out: as_quatern; track: as_ptrack; frame: single ): longint;
function as3d_getkeyhide( var out: longint; track: as_ptrack; frame: single ): longint;

function as3d_update: longint;

{$ifdef as_debug}
procedure as3d_debugworld;
{$endif}

// ==========================================================================

implementation

// --------------------------------------------------------------------------

procedure as_calcobjnormals( obj: as_pobject );
var
  a, b, normal: as_vector;
  i, j, num: longint;
begin

  for i := 0 to obj^.numfaces-1 do
    begin
      as_vect_sub( a, obj^.faces[i].pa^.vert, obj^.faces[i].pb^.vert );
      as_vect_sub( b, obj^.faces[i].pb^.vert, obj^.faces[i].pc^.vert );
      as_vect_cross( normal, a, b );
      as_vect_normalize( obj^.faces[i].norm, normal );
    end;

  for i := 0 to obj^.numverts-1 do
    begin
      num := 0;
      as_vect_zero( normal );
      for j := 0 to obj^.numfaces-1 do
        begin
          if (( obj^.faces[j].a = i ) or ( obj^.faces[j].b = i ) or ( obj^.faces[j].c = i )) then
            begin
              as_vect_add( normal, normal, obj^.faces[j].norm );
              inc( num );
            end;
//        if ( num > 0 ) then
//          as_vect_scale( normal, normal, 1.0 / single( num ));
          as_vect_normalize( obj^.verts[i].norm, normal );
        end;
    end;

end;

// --------------------------------------------------------------------------

procedure as_recalcobjnormals( obj: as_pobject );
var
  a, b, normal: as_vector;
  i, j, num: longint;
begin

  for i := 0 to obj^.numfaces-1 do
    begin
      as_vect_sub( a, obj^.faces[i].pa^.pvert, obj^.faces[i].pb^.pvert );
      as_vect_sub( b, obj^.faces[i].pb^.pvert, obj^.faces[i].pc^.pvert );
      as_vect_cross( normal, a, b );
      as_vect_normalize( obj^.faces[i].pnorm, normal );
    end;

  for i := 0 to obj^.numverts-1 do
    begin
      num := 0;
      as_vect_zero( normal );
      for j := 0 to obj^.numfaces-1 do
        begin
          if (( obj^.faces[j].a = i ) or ( obj^.faces[j].b = i ) or ( obj^.faces[j].c = i )) then
            begin
              as_vect_add( normal, normal, obj^.faces[j].pnorm );
              inc( num );
            end;
//        if ( num > 0 ) then
//          as_vect_scale( normal, normal, 1.0 / single( num ));
          as_vect_normalize( obj^.verts[i].pnorm, normal );
        end;
    end;

end;

// --------------------------------------------------------------------------

procedure as3d_calcnormals;
var
  node: as_pwnode;
begin

  node := as3d_scene^.world;
  while ( node <> nil ) do
    begin
      if ( node^.typ = as_obj_object ) then
        if (( as_pobject( node^.obj )^.numfaces > 0 ) and ( as_pobject( node^.obj )^.numverts > 0 )) then
          as_calcobjnormals( as_pobject( node^.obj ));
      node := node^.next;
    end;

end;

// --------------------------------------------------------------------------

procedure as_dotransform;
var
  node: as_pwnode;
  obj: as_pobject;
  objmat, normat: as_matrix;
  i: longint;
begin
  node := as3d_scene^.world;
  while ( node <> nil ) do
    begin

      if ( node^.typ = as_obj_object ) then
        begin
          obj := as_pobject( node^.obj );
          if ( as3d_camera <> nil ) then
            as_matr_mul( objmat, as3d_camera^.matrix, obj^.matrix ) else
            as_matr_copy( objmat, obj^.matrix );
          as_matr_normalize( normat, objmat );
          if ((( as3d_flags and as_internalmorph ) > 0 ) and (( obj^.flags and as_obj_morph ) > 0 )) then
            begin
              // [todo]: morphing.
            end else begin
              for i := 0 to obj^.numverts-1 do
                as_matr_mulvect( obj^.verts[i].pvert, obj^.verts[i].vert, objmat );
              if (( as3d_flags and as_calcnormal ) > 0 ) then
                begin
                  for i := 0 to obj^.numverts-1 do
                    as_matr_mulnorm( obj^.verts[i].pnorm, obj^.verts[i].norm, normat );
                  for i := 0 to obj^.numfaces-1 do
                    as_matr_mulnorm( obj^.faces[i].pnorm, obj^.faces[i].norm, normat );
                end;
            end;
        end;

      node := node^.next;
    end;
end;

// --------------------------------------------------------------------------

procedure as_freetrack( track: as_ptrack );
var
  key, next: as_pkey;
begin
  if ( track = nil ) then
    exit;
  key := track^.keys;
  while ( key <> nil ) do
    begin
      next := key^.next;
      dispose( key );
      key := next;
    end;
  dispose( track );
end;

// --------------------------------------------------------------------------

function strucmp( s1, s2: string ): boolean;
var
  i: longint;
begin
  if ( length( s1 ) <> length( s2 )) then
    begin
      strucmp := false;
      exit;
    end;
  for i := 1 to length( s1 ) do
    if ( upcase( s1[i] ) <> upcase( s2[i] )) then
      begin
        strucmp := false;
        exit;
      end;
  strucmp := true;
end;

function fmod( x, y: single ): single;
begin
  if ( y>0 ) then
    while ( x>y ) do
      x := x-y;
  fmod := x;
end;

// ==========================================================================

function as3d_init( flags: longint ): longint;
begin
  as3d_scene := nil;
  as3d_camera := nil;
  as3d_flags := flags;
  as3d_init := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_done: longint;
begin
  as3d_scene := nil;
  as3d_flags := 0;
  as3d_done := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_geterror( code: longint ): string;
begin
  case ( code ) of
    as_err_ok               : as3d_geterror := 'Ok.';
    as_err_out_of_memory    : as3d_geterror := 'Out of memory.';
    as_err_file_not_found   : as3d_geterror := 'File not found.';
    as_err_invalid_file     : as3d_geterror := 'Invalid (corrupted?) file.';
    as_err_invalid_version  : as3d_geterror := 'Unsupported version.';
    as_err_invalid_format   : as3d_geterror := 'Unsupported file format.';
    as_err_invalid_frame    : as3d_geterror := 'Invalid frame number.';
    as_err_invalid_name     : as3d_geterror := 'Invalid object name.';
    as_err_no_keyframes     : as3d_geterror := 'No keyframer data.';
    as_err_no_scene_loaded  : as3d_geterror := 'No scene loaded.';
    as_err_null_pointer     : as3d_geterror := 'Null pointer assignment.';
    as_err_invalid_param    : as3d_geterror := 'Invalid parameter.';
    as_err_invalid_spline   : as3d_geterror := 'Less than 2 keys in spline.';
    as_err_singular_matrix  : as3d_geterror := 'Can''t inverse singular matrix.';
    as_err_invalid_object   : as3d_geterror := 'Invalid object id number.';
    as_err_object_exists    : as3d_geterror := 'Object already exists.';
    as_err_undefined        : as3d_geterror := 'Internal error.';
  else
    as3d_geterror := '(undefined) internal error.';
  end;
end;

// --------------------------------------------------------------------------

function as3d_getframes( var start, ends: single ): longint;
begin
  if ( as3d_scene = nil ) then
    begin
      as3d_getframes := as_err_no_scene_loaded;
      exit;
    end;
  if ( as3d_scene^.keyfr = nil ) then
    begin
      as3d_getframes := as_err_no_scene_loaded;
      exit;
    end;
  if ( as3d_scene^.frameend - as3d_scene^.framestart <= 0 ) then
    begin
      as3d_getframes := as_err_no_keyframes;
      exit;
    end;
  start := as3d_scene^.framestart;
  ends := as3d_scene^.frameend;
  as3d_getframes := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_getframe( var frame: single ): longint;
begin
  if ( as3d_scene = nil ) then
    begin
      as3d_getframe := as_err_no_scene_loaded;
      exit;
    end;
  if ( as3d_scene^.keyfr = nil ) then
    begin
      as3d_getframe := as_err_no_scene_loaded;
      exit;
    end;
  if ( as3d_scene^.frameend - as3d_scene^.framestart <= 0 ) then
    begin
      as3d_getframe := as_err_no_keyframes;
      exit;
    end;
  frame := as3d_scene^.frame;
  as3d_getframe := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_setframe( frame: single ): longint;
begin
  if ( as3d_scene = nil ) then
    begin
      as3d_setframe := as_err_no_scene_loaded;
      exit;
    end;
  if ( as3d_scene^.keyfr = nil ) then
    begin
      as3d_setframe := as_err_no_scene_loaded;
      exit;
    end;
  if ( as3d_scene^.frameend - as3d_scene^.framestart <= 0 ) then
    begin
      as3d_setframe := as_err_no_keyframes;
      exit;
    end;
  if (( as3d_scene^.framestart > frame ) or ( as3d_scene^.frameend <= frame )) then
    begin
      as3d_setframe := as_err_invalid_frame;
      exit;
    end;
  as3d_scene^.frame := frame;
  as3d_setframe := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_setactivescene( scene: as_pscene ): longint;
begin
  if ( scene = nil ) then
    begin
      as3d_setactivescene := as_err_null_pointer;
      exit;
    end;
  as3d_scene := scene;
  as3d_setactivescene := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_getactivescene( var scene: as_pscene ): longint;
begin
  scene := as3d_scene;
  as3d_getactivescene := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_setactivecamera( camera: as_pcamera ): longint;
begin
{
  if ( camera = nil ) then
    begin
      as3d_setactivecamera := as_err_null_pointer;
      exit;
    end;
}
  as3d_camera := camera;
  as3d_setactivecamera := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_getactivecamera( var camera: as_pcamera ): longint;
begin
  camera := as3d_camera;
  as3d_getactivecamera := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_byname( var node: as_pwnode; name: string ): longint;
begin
  if ( as3d_scene = nil ) then
    begin
      as3d_byname := as_err_no_scene_loaded;
      exit;
    end;
  if ( as3d_scene^.world = nil ) then
    begin
      as3d_byname := as_err_no_scene_loaded;
      exit;
    end;
  node := as3d_scene^.world;
  while ( node <> nil ) do
    begin
      case ( node^.typ ) of
        as_obj_camera:
          if ( strucmp( as_pcamera( node^.obj )^.fname, name )) then
            begin
              as3d_byname := as_err_ok;
              exit;
            end;
        as_obj_object:
          if ( strucmp( as_pobject( node^.obj )^.fname, name )) then
            begin
              as3d_byname := as_err_ok;
              exit;
            end;
        as_obj_light:
          if ( strucmp( as_plight( node^.obj )^.fname, name )) then
            begin
              as3d_byname := as_err_ok;
              exit;
            end;
        as_obj_material:
          if ( strucmp( as_pmaterial( node^.obj )^.fname, name )) then
            begin
              as3d_byname := as_err_ok;
              exit;
            end;
        as_obj_ambient:
          if ( strucmp( as_pambient( node^.obj )^.fname, name )) then
            begin
              as3d_byname := as_err_ok;
              exit;
            end;
        end;
      node := node^.next;
    end;
  as3d_byname := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_byid( var node: as_pwnode; id: longint ): longint;
begin
  if ( as3d_scene = nil ) then
    begin
      as3d_byid := as_err_no_scene_loaded;
      exit;
    end;
  if ( as3d_scene^.world = nil ) then
    begin
      as3d_byid := as_err_no_scene_loaded;
      exit;
    end;
  node := as3d_scene^.world;
  while ( node <> nil ) do
    begin
      case ( node^.typ ) of
        as_obj_camera:
          if ( as_pcamera( node^.obj )^.id = id ) then
            begin
              as3d_byid := as_err_ok;
              exit;
            end;
        as_obj_object:
          if ( as_pobject( node^.obj )^.id = id ) then
            begin
              as3d_byid := as_err_ok;
              exit;
            end;
        as_obj_light:
          if ( as_plight( node^.obj )^.id = id ) then
            begin
              as3d_byid := as_err_ok;
              exit;
            end;
        as_obj_material:
          if ( as_pmaterial( node^.obj )^.id = id ) then
            begin
              as3d_byid := as_err_ok;
              exit;
            end;
        as_obj_ambient:
          if ( as_pambient( node^.obj )^.id = id ) then
            begin
              as3d_byid := as_err_ok;
              exit;
            end;
        end;
      node := node^.next;
    end;
  as3d_byid := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_findfirst( var node: as_pwnode; attr: longint ): longint;
begin
  if ( as3d_scene = nil ) then
    begin
      as3d_findfirst := as_err_no_scene_loaded;
      exit;
    end;
  if ( as3d_scene^.world = nil ) then
    begin
      as3d_findfirst := as_err_no_scene_loaded;
      exit;
    end;
  node := as3d_scene^.world;
  while ( node <> nil ) do
    begin
      if (( node^.typ and attr ) > 0 ) then
        begin
          as3d_findfirst := as_err_ok;
          exit;
        end;
      node := node^.next;
    end;
  as3d_findfirst := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_findnext( var node: as_pwnode; attr: longint ): longint;
begin
  if ( as3d_scene = nil ) then
    begin
      as3d_findnext := as_err_no_scene_loaded;
      exit;
    end;
  if ( as3d_scene^.world = nil ) then
    begin
      as3d_findnext := as_err_no_scene_loaded;
      exit;
    end;
  node := node^.next;
  while ( node <> nil ) do
    begin
      if (( node^.typ and attr ) > 0 ) then
        begin
          as3d_findnext := as_err_ok;
          exit;
        end;
      node := node^.next;
    end;
  as3d_findnext := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_addworld( typ: longint; obj: pointer ): longint;
var
  node: as_pwnode;
begin
  if ( as3d_scene = nil ) then
    begin
      as3d_addworld := as_err_no_scene_loaded;
      exit;
    end;
  new( node );
  if ( node = nil ) then
    begin
      as3d_addworld := as_err_out_of_memory;
      exit;
    end;
  node^.typ := typ;
  node^.obj := obj;
  node^.next := nil;
  if ( as3d_scene^.world = nil ) then
    begin
      node^.prev := nil;
      as3d_scene^.world := node;
      as3d_scene^.wtail := node;
    end else begin
      node^.prev := as3d_scene^.wtail;
      as3d_scene^.wtail^.next := node;
      as3d_scene^.wtail := node;
    end;
  as3d_addworld := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_addtrack( typ, id, parent: longint; track, obj: pointer ): longint;
var
  node, pnode: as_pknode;
begin
  if ( as3d_scene = nil ) then
    begin
      as3d_addtrack := as_err_no_scene_loaded;
      exit;
    end;
  new( node );
  if ( node = nil ) then
    begin
      as3d_addtrack := as_err_out_of_memory;
      exit;
    end;
  node^.typ := typ;
  node^.id := id;
  node^.track := track;
  node^.parent := nil;
  node^.brother := nil;
  node^.child := nil;
  node^.next := nil;
  node^.obj := obj;
  if ( as3d_scene^.keyfr = nil ) then
    begin
      node^.prev := nil;
      as3d_scene^.keyfr := node;
      as3d_scene^.ktail := node;
    end else begin
      node^.prev := as3d_scene^.ktail;
      as3d_scene^.ktail^.next := node;
      as3d_scene^.ktail := node;
    end;
  if ( parent <> -1 ) then
    begin
      pnode := as3d_scene^.keyfr;
      while ( pnode <> nil ) do
        begin
          if ( pnode^.id = parent ) then
            begin
              node^.parent := pnode;
              node^.brother := pnode^.child;
              pnode^.child := node;
            end;
          pnode := pnode^.next;
        end;
    end;
  as3d_addtrack := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_settrack( typ, id: longint; track: as_ptrack ): longint;
var
  node: as_pknode;
  obj: pointer;
begin
  if ( as3d_scene = nil ) then
    begin
      as3d_settrack := as_err_no_scene_loaded;
      exit;
    end;
  if ( as3d_scene^.keyfr = nil ) then
    begin
      as3d_settrack := as_err_no_scene_loaded;
      exit;
    end;
  node := as3d_scene^.keyfr;
  while (( node <> nil ) and ( node^.id <> id )) do
    node := node^.next;
  if ( node = nil ) then
    begin
      as3d_settrack := as_err_invalid_keyid;
      exit;
    end;
  obj := node^.track;
  case ( node^.typ ) of
    as_trk_camera:
      case ( typ ) of
        as_key_position:
          as_pcameratrack( obj )^.position := track;
        as_key_fov:
          as_pcameratrack( obj )^.fov := track;
        as_key_roll:
          as_pcameratrack( obj )^.roll := track;
        end;
    as_trk_camtarget:
      case ( typ ) of
        as_key_position:
          as_pcamtargettrack( obj )^.position := track;
        end;
    as_trk_omnilight:
      case ( typ ) of
        as_key_position:
          as_pomnilighttrack( obj )^.position := track;
        as_key_color:
          as_pomnilighttrack( obj )^.color := track;
        end;
    as_trk_spotlight:
      case ( typ ) of
        as_key_position:
          as_pspotlighttrack( obj )^.position := track;
        as_key_color:
          as_pspotlighttrack( obj )^.color := track;
        as_key_roll:
          as_pspotlighttrack( obj )^.roll := track;
        end;
    as_trk_spottarget:
      case ( typ ) of
        as_key_position:
          as_pspotlighttrack( obj )^.position := track;
        end;
    as_trk_object:
      case ( typ ) of
        as_key_position:
          as_pobjecttrack( obj )^.translate := track;
        as_key_scale:
          as_pobjecttrack( obj )^.scale := track;
        as_key_rotate:
          as_pobjecttrack( obj )^.rotate := track;
        as_key_morph:
          as_pobjecttrack( obj )^.morph := track;
        as_key_hide:
          as_pobjecttrack( obj )^.hide := track;
        end;
    as_trk_ambient:
      case ( typ ) of
        as_key_color:
          as_pambienttrack( obj )^.color := track;
        end;
    end;
  as3d_settrack := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_allocscene( var scene: as_pscene ): longint;
begin
  new( scene );
  if ( scene = nil ) then
    begin
      as3d_allocscene := as_err_out_of_memory;
      exit;
    end;
  scene^.world := nil;
  scene^.wtail := nil;
  scene^.keyfr := nil;
  scene^.ktail := nil;
  as3d_allocscene := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_freeworld( scene: as_pscene ): longint;
var
  node, next: as_pwnode;
  obj: as_pobject;
begin
  if ( scene = nil ) then
    begin
      as3d_freeworld := as_err_null_pointer;
      exit;
    end;
  if ( scene^.world = nil ) then
    begin
      as3d_freeworld := as_err_null_pointer;
      exit;
    end;
  node := scene^.world;
  while ( node <> nil ) do
    begin
      if ( node^.typ = as_obj_object ) then
        begin
          obj := as_pobject( node^.obj );
          freemem( obj^.verts, obj^.numverts * sizeof( as_vertex ));
          freemem( obj^.faces, obj^.numfaces * sizeof( as_face ));
          dispose( obj );
        end;
      next := node^.next;
      dispose( node );
      node := next;
    end;
  scene^.world := nil;
  scene^.wtail := nil;
  as3d_freeworld := as_err_ok;
end;

// --------------------------------------------------------------------------

function as3d_freemotion( scene: as_pscene ): longint;
var
  node, next: as_pknode;
  cam: as_pcameratrack;
  ctrg: as_pcamtargettrack;
  omni: as_pomnilighttrack;
  spot: as_pspotlighttrack;
  strg: as_pspottargettrack;
  obj: as_pobjecttrack;
  amb: as_pambienttrack;
begin
  if ( scene = nil ) then
    begin
      as3d_freemotion := as_err_null_pointer;
      exit;
    end;
  if ( scene^.world = nil ) then
    begin
      as3d_freemotion := as_err_null_pointer;
      exit;
    end;
  node := scene^.keyfr;
  while ( node <> nil ) do
    begin
      next := node^.next;
      case ( node^.typ ) of
        as_trk_camera:
          begin
            cam := as_pcameratrack( node^.track );
            as_freetrack( cam^.position );
            as_freetrack( cam^.fov );
            as_freetrack( cam^.roll );
            dispose( cam );
          end;
        as_trk_camtarget:
          begin
            ctrg := as_pcamtargettrack( node^.track );
            as_freetrack( ctrg^.position );
            dispose( ctrg );
          end;
        as_trk_omnilight:
          begin
            omni := as_pomnilighttrack( node^.track );
            as_freetrack( omni^.position );
            as_freetrack( omni^.color );
            dispose( omni );
          end;
        as_trk_spotlight:
          begin
            spot := as_pspotlighttrack( node^.track );
            as_freetrack( spot^.position );
            as_freetrack( spot^.color );
            as_freetrack( spot^.roll );
            dispose( spot );
          end;
        as_trk_spottarget:
          begin
            strg := as_pspottargettrack( node^.track );
            as_freetrack( strg^.position );
            dispose( strg );
          end;
        as_trk_object:
          begin
            obj := as_pobjecttrack( node^.track );
            as_freetrack( obj^.translate );
            as_freetrack( obj^.scale );
            as_freetrack( obj^.rotate );
            as_freetrack( obj^.morph );
            as_freetrack( obj^.hide );
            dispose( obj );
          end;
        as_trk_ambient:
          begin
            amb := as_pambienttrack( node^.track );
            as_freetrack( amb^.color );
            dispose( amb );
          end;
        end;
      dispose( node );
      node := next;
    end;
  scene^.keyfr := nil;
  scene^.ktail := nil;
  as3d_freemotion := as_err_ok;
end;

// --------------------------------------------------------------------------

// [TODO]: as3d_freemesh()

// --------------------------------------------------------------------------

function as3d_freescene( scene: as_pscene ): longint;
var
  err: longint;
begin
  err := as3d_freemotion( scene );
  if ( err <> as_err_ok ) then
    begin
      as3d_freescene := err;
      exit;
    end;
  as3d_freescene := as3d_freeworld( scene );
end;

// --------------------------------------------------------------------------

// [TODO]: Debug functions (printworld/printkeyframer)

{$ifdef as_debug }

procedure as3d_debugworld;
var
  node: as_pwnode;
  obj: as_pobject;
  cam: as_pcamera;
  t: text;
  i: longint;
begin
  assign( t, 'world3ds' );
  rewrite( t );
  node := as3d_scene^.world;
  while ( node <> nil ) do
    begin
      write( t, 'NODE[', node^.typ, ']: ' );
      case ( node^.typ ) of
        as_obj_object:
          begin
            obj := as_pobject( node^.obj );
            writeln( t, 'object #', obj^.id, ' "', obj^.fname, '"' );
            writeln( t, '  parent #', obj^.parent );
            writeln( t, '  matrix = ' );
            writeln( t, '    xx:', obj^.matrix[ 0, 0 ]:9:3, ' xy:', obj^.matrix[ 0, 1 ]:9:3, ' xz:', obj^.matrix[ 0, 2 ]:9:3, ' xw:', obj^.matrix[ 0, 3 ]:9:3 );
            writeln( t, '    yx:', obj^.matrix[ 1, 0 ]:9:3, ' yy:', obj^.matrix[ 1, 1 ]:9:3, ' yz:', obj^.matrix[ 1, 2 ]:9:3, ' yw:', obj^.matrix[ 1, 3 ]:9:3 );
            writeln( t, '    zx:', obj^.matrix[ 2, 0 ]:9:3, ' zy:', obj^.matrix[ 2, 1 ]:9:3, ' zz:', obj^.matrix[ 2, 2 ]:9:3, ' zw:', obj^.matrix[ 2, 3 ]:9:3 );
            writeln( t, '  verts [', obj^.numverts, '] = ' );
            for i := 0 to obj^.numverts-1 do
              writeln( t, '    x:', obj^.verts[i].vert.x:9:3, ' y:', obj^.verts[i].vert.y:9:3, ' z:', obj^.verts[i].vert.z:9:3 );
          end;
        as_obj_camera:
          begin
            cam := as_pcamera( node^.obj );
            writeln( t, 'camera #', cam^.id, ' "', cam^.fname, '"' );
            writeln( t, '  parent #', cam^.parent1, ' #', cam^.parent2 );
            writeln( t, '  position = ' );
            writeln( t, '    x:', cam^.position.x:9:3, ' y:', cam^.position.y:9:3, ' z:', cam^.position.z:9:3 );
            writeln( t, '  target = ' );
            writeln( t, '    x:', cam^.target.x:9:3, ' y:', cam^.target.y:9:3, ' z:', cam^.target.z:9:3 );
            writeln( t, '  matrix = ' );
            writeln( t, '    xx:', cam^.matrix[ 0, 0 ]:9:3, ' xy:', cam^.matrix[ 0, 1 ]:9:3, ' xz:', cam^.matrix[ 0, 2 ]:9:3, ' xw:', cam^.matrix[ 0, 3 ]:9:3 );
            writeln( t, '    yx:', cam^.matrix[ 1, 0 ]:9:3, ' yy:', cam^.matrix[ 1, 1 ]:9:3, ' yz:', cam^.matrix[ 1, 2 ]:9:3, ' yw:', cam^.matrix[ 1, 3 ]:9:3 );
            writeln( t, '    zx:', cam^.matrix[ 2, 0 ]:9:3, ' zy:', cam^.matrix[ 2, 1 ]:9:3, ' zz:', cam^.matrix[ 2, 2 ]:9:3, ' zw:', cam^.matrix[ 2, 3 ]:9:3 );
          end;
        else
          writeln( t, 'Unknown.' );
        end;
      node := node^.next;
    end;
  close( t );
end;

{$endif}

// --------------------------------------------------------------------------

function as3d_getkeyfloat( var out: single; track: as_ptrack; frame: single ): longint;
var
  keys: as_pkey;
begin
  if ( frame < 0.0 ) then
    begin
      as3d_getkeyfloat := as_err_invalid_frame;
      exit;
    end;
  if ( track = nil ) then
    begin
      as3d_getkeyfloat := as_err_null_pointer;
      exit;
    end;
  if ( track^.keys = nil ) then
    begin
      as3d_getkeyfloat := as_err_null_pointer;
      exit;
    end;
  keys := track^.keys;
  if ( track^.flags <> 0 ) then
    frame := fmod( frame, track^.frames );
  if (( keys^.next = nil ) or ( frame < keys^.frame )) then
    begin
      out := keys^.val.w;
      as3d_getkeyfloat := as_err_ok;
    end;
  as3d_getkeyfloat := as_spline_getkeyfloat( out, track, frame );
end;

// --------------------------------------------------------------------------

function as3d_getkeyvect( var out: as_vector; track: as_ptrack; frame: single ): longint;
var
  keys: as_pkey;
begin
  if ( frame < 0.0 ) then
    begin
      as3d_getkeyvect := as_err_invalid_frame;
      exit;
    end;
  if ( track = nil ) then
    begin
      as3d_getkeyvect := as_err_null_pointer;
      exit;
    end;
  if ( track^.keys = nil ) then
    begin
      as3d_getkeyvect := as_err_null_pointer;
      exit;
    end;
  keys := track^.keys;
  if ( track^.flags <> 0 ) then
    frame := fmod( frame, track^.frames );
  if (( keys^.next = nil ) or ( frame < keys^.frame )) then
    begin
      out.x := keys^.val.x;
      out.y := keys^.val.y;
      out.z := keys^.val.z;
      as3d_getkeyvect := as_err_ok;
    end;
  as3d_getkeyvect := as_spline_getkeyvect( out, track, frame );
end;

// --------------------------------------------------------------------------

function as3d_getkeycolor( var out: as_color; track: as_ptrack; frame: single ): longint;
var
  vect: as_vector;
  err: longint;
begin
  err := as3d_getkeyvect( vect, track, frame );
  if ( err = as_err_ok ) then
    begin
      out.r := vect.x;
      out.g := vect.y;
      out.b := vect.z;
    end;
  as3d_getkeycolor := err;
end;

// --------------------------------------------------------------------------

function as3d_getkeyquat( var out: as_quatern; track: as_ptrack; frame: single ): longint;
var
  keys: as_pkey;
begin
  if ( frame < 0.0 ) then
    begin
      as3d_getkeyquat := as_err_invalid_frame;
      exit;
    end;
  if ( track = nil ) then
    begin
      as3d_getkeyquat := as_err_null_pointer;
      exit;
    end;
  if ( track^.keys = nil ) then
    begin
      as3d_getkeyquat := as_err_null_pointer;
      exit;
    end;
  keys := track^.keys;
  if ( track^.flags <> 0 ) then
    frame := fmod( frame, track^.frames );
  if (( keys^.next = nil ) or ( frame < keys^.frame )) then
    begin
      as_quat_copy( out, keys^.val );
      as3d_getkeyquat := as_err_ok;
    end;
  as3d_getkeyquat := as_spline_getkeyquat( out, track, frame );
end;

// --------------------------------------------------------------------------

function as3d_getkeyhide( var out: longint; track: as_ptrack; frame: single ): longint;
var
  keys: as_pkey;
begin
  if ( frame < 0.0 ) then
    begin
      as3d_getkeyhide := as_err_invalid_frame;
      exit;
    end;
  if ( track = nil ) then
    begin
      as3d_getkeyhide := as_err_null_pointer;
      exit;
    end;
  if ( track^.keys = nil ) then
    begin
      as3d_getkeyhide := as_err_null_pointer;
      exit;
    end;
  keys := track^.keys;
  if ( track^.flags <> 0 ) then
    frame := fmod( frame, track^.frames );
  if ( frame < track^.last^.frame ) then
    keys := track^.last;

  while (( keys^.next <> nil ) and ( frame > keys^.next^.frame )) do
    keys := keys^.next;
  track^.last := keys;     // [?]
  out := round( keys^.val.w );
  as3d_getkeyhide := as_err_ok;
end;

// --------------------------------------------------------------------------

// [TODO]: as3d_getkeymorph().

// --------------------------------------------------------------------------

function as3d_update: longint;
var
  node, child: as_pknode;
  cam: as_pcamera;
  tcam: as_pcameratrack;
  tctrg: as_pcamtargettrack;
  lit: as_plight;
  tomni: as_pomnilighttrack;
  tspot: as_pspotlighttrack;
  tstrg: as_pspottargettrack;
  amb: as_pambient;
  tamb: as_pambienttrack;
  obj, cobj: as_pobject;
  tobj: as_pobjecttrack;
  c, d: as_matrix;
  hidden: longint;
  frame: single;
begin

  if ( as3d_scene = nil ) then
    begin
      as3d_update := as_err_no_scene_loaded;
      exit;
    end;
  if ( as3d_scene^.world = nil ) then
    begin
      as3d_update := as_err_no_scene_loaded;
      exit;
    end;
  if ( as3d_scene^.keyfr = nil ) then
    begin
      as3d_update := as_err_no_keyframes;
      exit;
    end;

  frame := as3d_scene^.frame;

  // --- update keyframer data.

  node := as3d_scene^.keyfr;
  while ( node <> nil ) do
    begin
      case ( node^.typ ) of
        as_trk_camera:
          begin
            cam := as_pcamera( node^.obj );
            tcam := as_pcameratrack( node^.track );
            as3d_getkeyvect( cam^.position, tcam^.position, frame );
            as3d_getkeyfloat( cam^.fov, tcam^.fov, frame );
            as3d_getkeyfloat( cam^.roll, tcam^.roll, frame );
          end;
        as_trk_camtarget:
          begin
            cam := as_pcamera( node^.obj );
            tctrg := as_pcamtargettrack( node^.track );
            as3d_getkeyvect( cam^.target, tctrg^.position, frame );
          end;
        as_trk_omnilight:
          begin
            lit := as_plight( node^.obj );
            tomni := as_pomnilighttrack( node^.track );
            as3d_getkeyvect( lit^.position, tomni^.position, frame );
            as3d_getkeycolor( lit^.color, tomni^.color, frame );
          end;
        as_trk_spotlight:
          begin
            lit := as_plight( node^.obj );
            tspot := as_pspotlighttrack( node^.track );
            as3d_getkeyvect( lit^.position, tspot^.position, frame );
            as3d_getkeycolor( lit^.color, tspot^.color, frame );
            as3d_getkeyfloat( lit^.roll, tspot^.roll, frame );
          end;
        as_trk_spottarget:
          begin
            lit := as_plight( node^.obj );
            tstrg := as_pspottargettrack( node^.track );
            as3d_getkeyvect( lit^.target, tstrg^.position, frame );
          end;
        as_trk_object:
          begin
            obj := as_pobject( node^.obj );
            tobj := as_pobjecttrack( node^.track );
            as3d_getkeyvect( obj^.translate, tobj^.translate, frame );
            as3d_getkeyvect( obj^.scale, tobj^.scale, frame );
            as3d_getkeyquat( obj^.rotate, tobj^.rotate, frame );
            as3d_getkeyhide( hidden, tobj^.hide, frame );
            if ( hidden <> 0 ) then
              obj^.flags := obj^.flags or as_obj_hidden else
              obj^.flags := obj^.flags and ( not as_obj_hidden );
// [TODO]:  if ( as3d_getkeymorph( obj^.morph, tobj^.morph, frame ) = as_err_ok ) then
//            obj^.flags := obj^.flags or as_obj_morph else
              obj^.flags := obj^.flags and ( not as_obj_morph );
            as_quat_toinvmatrix( c, obj^.rotate );
            as_matr_setscale( d, obj^.scale );
            c[ 0, 3 ] := obj^.translate.x;
            c[ 1, 3 ] := obj^.translate.y;
            c[ 2, 3 ] := obj^.translate.z;
            as_matr_mul( obj^.matrix, c, d );
          end;
        as_trk_ambient:
          begin
            amb := as_pambient( node^.obj );
            tamb := as_pambienttrack( node^.track );
            as3d_getkeycolor( amb^.color, tamb^.color, frame );
          end;
        end;
      node := node^.next;
    end;

  // --- update hierarchy.
{
  if (( as3d_flags and as_hierarchy ) > 0 ) then
    begin
      node := as3d_scene^.keyfr;
      while ( node <> nil ) do
        begin
          if ( node^.typ = as_trk_object ) then
            begin
              obj := as_pobject( node^.obj );
              child := node^.child;
              while ( child <> nil ) do
                begin
                  case ( child^.typ ) of
                    as_trk_object:
                      begin
                        cobj := as_pobject( child^.obj );
                        as_matr_mul( cobj^.matrix, obj^.matrix, cobj^.matrix );
                      end;
                    as_trk_camera:
                      begin
                        cam := as_pcamera( child^.obj );
                        as_matr_mulvect( cam^.position, cam^.position, obj^.matrix );
                      end;
                    as_trk_camtarget:
                      begin
                        cam := as_pcamera( child^.obj );
                        as_matr_mulvect( cam^.target, cam^.target, obj^.matrix );
                      end;
                    as_trk_omnilight, as_trk_spotlight:
                      begin
                        lit := as_plight( child^.obj );
                        as_matr_mulvect( lit^.position, lit^.position, obj^.matrix );
                      end;
                    as_trk_spottarget:
                      begin
                        lit := as_plight( child^.obj );
                        as_matr_mulvect( lit^.target, lit^.target, obj^.matrix );
                      end;
                    end;
                  child := child^.brother;
                end;
            end;
          if ( node^.typ = as_trk_camera ) then
            begin
              cam := as_pcamera( node^.obj );
              child := node^.child;
              while ( child <> nil ) do
                begin
                  case ( child^.typ ) of
                    as_trk_omnilight, as_trk_spotlight:
                      begin
                        lit := as_plight( child^.obj );
                        as_vect_add( lit^.position, cam^.position, lit^.position );
                      end;
                    end;
                  child := child^.brother;
                end;
            end;
          node := node^.next;
        end;
    end;
}
  // --- update camera matrices.

  node := as3d_scene^.keyfr;
  while ( node <> nil ) do
    begin
      if ( node^.typ = as_trk_camera ) then
        begin
          cam := as_pcamera( node^.obj );
          as_cam_update( cam );
        end;
      node := node^.next;
    end;

  if (( as3d_flags and as_transform ) > 0 ) then
    as_dotransform;

  as3d_update := as_err_ok;

end;

end.