
unit as_vect;

// ==========================================================================
//  autospace3d - version 0.90 / o1.o8.2ooo
//  freepascal (clax) keyframing & (opengl) 3d engine - vector math.
// --------------------------------------------------------------------------
//  copyright (c)2ooo. remage / fresh!mindworkz.
// --------------------------------------------------------------------------

interface

uses 
  math, as_type;

// --------------------------------------------------------------------------

procedure as_vect_zero( var out: as_vector );
procedure as_vect_make( var out: as_vector; x, y, z: single );
procedure as_vect_copy( var out: as_vector; in1: as_vector );

procedure as_vect_add( var out: as_vector; in1, in2: as_vector );
procedure as_vect_sub( var out: as_vector; in1, in2: as_vector );
procedure as_vect_mul( var out: as_vector; in1, in2: as_vector );

procedure as_vect_scale( var out: as_vector; in1: as_vector; s: single );
procedure as_vect_rescale( var out: as_vector; in1: as_vector; s: single );
procedure as_vect_negate( var out: as_vector; in1: as_vector );

function as_vect_equal( in1, in2: as_vector ): boolean;

function as_vect_length( in1: as_vector ): single;
function as_vect_distance( in1, in2: as_vector ): single;

function as_vect_dot( in1, in2: as_vector ): single;
function as_vect_dotunit( in1, in2: as_vector ): single;

procedure as_vect_cross( var out: as_vector; in1, in2: as_vector );

procedure as_vect_midpoint( var out: as_vector; in1, in2: as_vector );
procedure as_vect_lerp( var out: as_vector; in1, in2: as_vector; alpha: single );
procedure as_vect_combine( var out: as_vector; in1, in2: as_vector; as, bs: single );

procedure as_vect_normalize( var out: as_vector; in1: as_vector );

// ==========================================================================

implementation

// --- vect_zero ------------------------------------------------------------

procedure as_vect_zero( var out: as_vector );
begin
  out.x := 0;
  out.y := 0;
  out.z := 0;
end;

// --- vect_make ------------------------------------------------------------

procedure as_vect_make( var out: as_vector; x, y, z: single );
begin
  out.x := x;
  out.y := y;
  out.z := z;
end;

// --- vect_copy ------------------------------------------------------------

procedure as_vect_copy( var out: as_vector; in1: as_vector );
begin
  out.x := in1.x;
  out.y := in1.y;
  out.z := in1.z;
end;

// --- vect_print -----------------------------------------------------------

procedure as_vect_print( in1: as_vector );
begin
  writeln( ' x: ', in1.x:9:3, ' y: ', in1.y:9:3, ' z: ', in1.z:9:3 );
end;

// --- vect_add -------------------------------------------------------------

procedure as_vect_add( var out: as_vector; in1, in2: as_vector );
var
  tmp: as_vector;
begin
  tmp.x := in1.x + in2.x;
  tmp.y := in1.y + in2.y;
  tmp.z := in1.z + in2.z;
  as_vect_copy( out, tmp );
end;

// --- vect_sub -------------------------------------------------------------

procedure as_vect_sub( var out: as_vector; in1, in2: as_vector );
var
  tmp: as_vector;
begin
  tmp.x := in1.x - in2.x;
  tmp.y := in1.y - in2.y;
  tmp.z := in1.z - in2.z;
  as_vect_copy( out, tmp );
end;

// --- vect_mul -------------------------------------------------------------

procedure as_vect_mul( var out: as_vector; in1, in2: as_vector );
var
  tmp: as_vector;
begin
  tmp.x := in1.x * in2.x;
  tmp.y := in1.y * in2.y;
  tmp.z := in1.z * in2.z;
  as_vect_copy( out, tmp );
end;

// --- vect_scale -----------------------------------------------------------

procedure as_vect_scale( var out: as_vector; in1: as_vector; s: single );
var
  tmp: as_vector;
begin
  tmp.x := in1.x * s;  
  tmp.y := in1.y * s;  
  tmp.z := in1.z * s;  
  as_vect_copy( out, tmp );
end;

// --- vect_rescale ---------------------------------------------------------

procedure as_vect_rescale( var out: as_vector; in1: as_vector; s: single );
var
  tmp: as_vector;
  len: single;
begin
  len := sqrt( sqr( in1.x ) + sqr( in1.y ) + sqr( in1.z ));
  if ( len = 0 ) then
    begin
      out.x := 0;
      out.y := 0;
      out.z := 0;
    end else begin
      s := s / len;
      tmp.x := in1.x * s;  
      tmp.y := in1.y * s;  
      tmp.z := in1.z * s;  
      as_vect_copy( out, tmp );
    end;
end;

// --- vect_negate ----------------------------------------------------------

procedure as_vect_negate( var out: as_vector; in1: as_vector );
var
  tmp: as_vector;
begin
  tmp.x := - in1.x;
  tmp.y := - in1.y;
  tmp.z := - in1.z;
  as_vect_copy( out, tmp );
end;

// --- vect_equal -----------------------------------------------------------

function as_vect_equal( in1, in2: as_vector ): boolean;
begin
  as_vect_equal := ( in1.x = in2.x ) and ( in1.y = in2.y ) and ( in1.z = in2.z );
end;

// --- vect_length ----------------------------------------------------------

function as_vect_length( in1: as_vector ): single;
var
  len: single;
begin
  len := sqrt( sqr( in1.x ) + sqr( in1.y ) + sqr( in1.z ));
  if ( len = 0.0 ) then len := 1.0;
  as_vect_length := len;
end;

// --- vect_distance --------------------------------------------------------

function as_vect_distance( in1, in2: as_vector ): single;
var
  tmp: as_vector;
begin
  as_vect_sub( tmp, in1, in2 );
  as_vect_distance := as_vect_length( tmp );
end;

// --- vect_dot -------------------------------------------------------------

function as_vect_dot( in1, in2: as_vector ): single;
var
  mag: single;
begin
  mag := as_vect_length( in1 ) * as_vect_length( in2 );
  as_vect_dot := ( in1.x*in2.x + in1.y*in2.y + in1.z*in2.z ) / mag;
end;

// --- vect_dotunit ---------------------------------------------------------

function as_vect_dotunit( in1, in2: as_vector ): single;
begin
  as_vect_dotunit := in1.x*in2.x + in1.y*in2.y + in1.z*in2.z;
end;

// --- vect_cross -----------------------------------------------------------

procedure as_vect_cross( var out: as_vector; in1, in2: as_vector );
var
  tmp: as_vector;
begin
  tmp.x := in1.y*in2.z - in1.z*in2.y;
  tmp.y := in1.z*in2.x - in1.x*in2.z;
  tmp.z := in1.x*in2.y - in1.y*in2.x;
  as_vect_copy( out, tmp );
end;

// --- vect_midpoint --------------------------------------------------------

procedure as_vect_midpoint( var out: as_vector; in1, in2: as_vector );
var
  tmp: as_vector;
begin
  as_vect_add( tmp, in1, in2 );
  as_vect_scale( tmp, tmp, 0.5 );
  as_vect_copy( out, tmp );
end;

// --- vect_lerp ------------------------------------------------------------

procedure as_vect_lerp( var out: as_vector; in1, in2: as_vector; alpha: single );
var
  tmp: as_vector;
begin
  tmp.x := in1.x + ( in2.x - in1.x ) * alpha;
  tmp.y := in1.y + ( in2.y - in1.y ) * alpha;
  tmp.z := in1.z + ( in2.z - in1.z ) * alpha;
  as_vect_copy( out, tmp );
end;

// --- vect_combine ---------------------------------------------------------

procedure as_vect_combine( var out: as_vector; in1, in2: as_vector; as, bs: single );
var
  tmp: as_vector;
begin
  tmp.x := in1.x * as + in2.x * bs;
  tmp.y := in1.y * as + in2.y * bs;
  tmp.z := in1.z * as + in2.z * bs;
  as_vect_copy( out, tmp );
end;

// --- vect_normalize -------------------------------------------------------

procedure as_vect_normalize( var out: as_vector; in1: as_vector );
var
  tmp: as_vector;
  len: single;
begin
  len := sqrt( sqr( in1.x ) + sqr( in1.y ) + sqr( in1.z ));
  if ( len = 0.0 ) then 
    begin
      out.x := 0;
      out.y := 0;
      out.z := 0;
    end else begin
      len := 1.0 / len;
      tmp.x := in1.x * len;
      tmp.y := in1.y * len;
      tmp.z := in1.z * len;
      as_vect_copy( out, tmp );      
    end;
end;

end.