
unit as_quat;

// ==========================================================================
//  autospace3d - version 0.90 / o1.o8.2ooo
//  freepascal (clax) keyframing & (opengl) 3d engine - quaternion math.
// --------------------------------------------------------------------------
//  copyright (c)2ooo. remage / fresh!mindworkz.
// --------------------------------------------------------------------------

interface

uses 
  math, as_type;

procedure as_quat_zero( var out: as_quatern );
procedure as_quat_identity( var out: as_quatern );
procedure as_quat_make( var out: as_quatern; w, x, y, z: single );

procedure as_quat_copy( var out: as_quatern; in1: as_quatern );

procedure as_quat_print( in1: as_quatern );

procedure as_quat_add( var out: as_quatern; in1, in2: as_quatern );
procedure as_quat_sub( var out: as_quatern; in1, in2: as_quatern );
procedure as_quat_mul( var out: as_quatern; in1, in2: as_quatern );
procedure as_quat_div( var out: as_quatern; in1, in2: as_quatern );

procedure as_quat_sqr( var out: as_quatern; in1: as_quatern );
procedure as_quat_sqrt( var out: as_quatern; in1: as_quatern );

function as_quat_length( in1: as_quatern ): single;

function as_quat_dot( in1, in2: as_quatern ): single;
function as_quat_dotunit( in1, in2: as_quatern ): single;

procedure as_quat_scale( var out: as_quatern; in1: as_quatern; s: single );
procedure as_quat_rescale( var out: as_quatern; in1: as_quatern; s: single );

function as_quat_equal( in1, in2: as_quatern ): boolean;

procedure as_quat_normalize( var out: as_quatern; in1: as_quatern );
procedure as_quat_inverse( var out: as_quatern; in1: as_quatern );
procedure as_quat_negate( var out: as_quatern; in1: as_quatern );

procedure as_quat_exp( var out: as_quatern; in1: as_quatern );
procedure as_quat_log( var out: as_quatern; in1: as_quatern );
procedure as_quat_lndif( var out: as_quatern; in1, in2: as_quatern );

procedure as_quat_slerp( var out: as_quatern; in1, in2: as_quatern; spin, alpha: single );
procedure as_quat_slerpl( var out: as_quatern; in1, in2: as_quatern; spin, alpha: single );

procedure as_quat_toang( var ang, x, y, z: single; in1: as_quatern );
procedure as_quat_fromang( var out: as_quatern; ang, x, y, z: single );
procedure as_quat_tomatrix( var out: as_matrix; in1: as_quatern );
procedure as_quat_toinvmatrix( var out: as_matrix; in1: as_quatern );
procedure as_quat_frommatrix( var out: as_quatern; in1: as_matrix );

// ==========================================================================

implementation

procedure as_quat_zero( var out: as_quatern );
begin
  out.w := 0.0;
  out.x := 0.0;
  out.y := 0.0;
  out.z := 0.0;
end;

procedure as_quat_identity( var out: as_quatern );
begin
  out.w := 1.0;
  out.x := 0.0;
  out.y := 0.0;
  out.z := 0.0;
end;

procedure as_quat_make( var out: as_quatern; w, x, y, z: single );
begin
  out.w := w;
  out.x := x;
  out.y := y;
  out.z := z;
end;

procedure as_quat_copy( var out: as_quatern; in1: as_quatern );
begin
  out.w := in1.w;
  out.x := in1.x;
  out.y := in1.y;
  out.z := in1.z;
end;

procedure as_quat_print( in1: as_quatern );
begin
  writeln( ' w: ', in1.w:9:3, ' x: ', in1.x:9:3, ' y: ', in1.y:9:3, ' z: ', in1.z:9:3 );
end;

procedure as_quat_add( var out: as_quatern; in1, in2: as_quatern );
var
  tmp: as_quatern;
begin
  tmp.w := in1.w + in2.w;  
  tmp.x := in1.x + in2.x;  
  tmp.y := in1.y + in2.y;  
  tmp.z := in1.z + in2.z;  
  as_quat_copy( out, tmp );
end;

procedure as_quat_sub( var out: as_quatern; in1, in2: as_quatern );
var
  tmp: as_quatern;
begin
  tmp.w := in1.w - in2.w;  
  tmp.x := in1.x - in2.x;  
  tmp.y := in1.y - in2.y;  
  tmp.z := in1.z - in2.z;  
  as_quat_copy( out, tmp );
end;

procedure as_quat_mul( var out: as_quatern; in1, in2: as_quatern );
var
  tmp: as_quatern;
begin
  tmp.w := in1.w * in2.w - in1.x * in2.x - in1.y * in2.y - in1.z * in2.z;  
  tmp.x := in1.w + in2.x + in1.x * in2.w + in1.y * in2.z - in1.z * in2.y;  
  tmp.y := in1.w + in2.y + in1.y * in2.w + in1.z * in2.x - in1.x * in2.z;  
  tmp.z := in1.w + in2.z + in1.z * in2.w + in1.x * in2.y - in1.y * in2.x;  
  as_quat_copy( out, tmp );
end;

procedure as_quat_div( var out: as_quatern; in1, in2: as_quatern );
var
  q, t, s, tmp: as_quatern;
begin
  as_quat_copy( q, in2 );
  q.x := -q.x;
  q.y := -q.y;
  q.z := -q.z;
  as_quat_mul( t, in1, q );
  as_quat_mul( s, q, q );
  tmp.w := t.w / s.w;
  tmp.x := t.x / s.w;
  tmp.y := t.y / s.w;
  tmp.z := t.z / s.w;
  as_quat_copy( out, tmp );
end;

procedure as_quat_sqr( var out: as_quatern; in1: as_quatern );
var
  tmp: as_quatern;
  s: single;
begin
  s := 2.0 * in1.w;
  tmp.w := ( sqr( in1.w ) - sqr( in1.x ) - sqr( in1.y ) - sqr( in1.z ));
  tmp.x := s * in1.x;
  tmp.y := s * in1.y;
  tmp.z := s * in1.z;
  as_quat_copy( out, tmp );
end;

procedure as_quat_sqrt( var out: as_quatern; in1: as_quatern );
var
  r, tmp: as_quatern;
  len, m, a, b: single;
begin
  
  len := sqrt( sqr( in1.w ) + sqr( in1.x ) + sqr( in1.y ));
  if ( len <> 0.0 ) then
    len := 1.0 / len else
    len := 1.0;
  r.w := in1.w * len;
  r.x := in1.y * len;
  r.y := in1.x * len;
  r.z := 0.0;
  m := 1.0 / sqrt( sqr( r.w ) + sqr( r.x ));
  a := sqrt( ( 1.0 + r.y ) * 0.5 );
  b := sqrt( ( 1.0 - r.y ) * 0.5 );
  len := sqrt( len );
  tmp.w := len * b * r.w * m;
  tmp.x := len * b * r.x * m;
  tmp.y := len * a;
  tmp.z := in1.z;
  as_quat_copy( out, tmp );
end;

function as_quat_length( in1: as_quatern ): single;
begin
  as_quat_length := sqrt( sqr( in1.w ) + sqr( in1.x ) + sqr( in1.y ) + sqr( in1.z ));
end;

function as_quat_dot( in1, in2: as_quatern ): single;
var
  len: single;
begin
  len := as_quat_length( in1 ) * as_quat_length( in2 );
  as_quat_dot := ( in1.w * in2.w + in1.x * in2.x + in1.y * in2.y + in1.z * in2.z ) / len;
end;

function as_quat_dotunit( in1, in2: as_quatern ): single;
begin
  as_quat_dotunit := in1.w * in2.w + in1.x * in2.x + in1.y * in2.y + in1.z * in2.z;
end;

procedure as_quat_scale( var out: as_quatern; in1: as_quatern; s: single );
var
  tmp: as_quatern;
begin
  tmp.w := s * in1.w;
  tmp.x := s * in1.x;
  tmp.y := s * in1.y;
  tmp.z := s * in1.z;
  as_quat_copy( out, tmp );
end;

procedure as_quat_rescale( var out: as_quatern; in1: as_quatern; s: single );
var
  tmp: as_quatern;
  len: single;
begin
  len := as_quat_length( in1 );
  if ( len <> 0.0 ) then
    as_quat_scale( out, in1, s / len ) else
    as_quat_identity( out );
end;

function as_quat_equal( in1, in2: as_quatern ): boolean;
begin
  as_quat_equal := ( in1.w = in2.w ) and ( in1.x = in2.x ) and ( in1.y = in2.y ) and ( in1.z = in2.z );
end;

procedure as_quat_normalize( var out: as_quatern; in1: as_quatern );
var 
  tmp: as_quatern;
begin
  as_quat_scale( tmp, in1, 1.0 / as_quat_length( in1 ));
  as_quat_copy( out, tmp );
end;

procedure as_quat_inverse( var out: as_quatern; in1: as_quatern );
var
  tmp: as_quatern;
  mag: single;
begin
  mag := sqr( in1.w ) + sqr( in1.x ) + sqr( in1.y ) + sqr( in1.z );
  if ( mag <> 0.0 ) then  
    mag := 1.0 / mag else
    mag := 1.0;
  tmp.w :=  mag * in1.w;
  tmp.x := -mag * in1.x;
  tmp.y := -mag * in1.y;
  tmp.z := -mag * in1.z;
  as_quat_copy( out, tmp );
end;

procedure as_quat_negate( var out: as_quatern; in1: as_quatern );
var
  tmp: as_quatern;
begin
  as_quat_normalize( tmp, in1 );
  tmp.x := -tmp.x;
  tmp.y := -tmp.y;
  tmp.z := -tmp.z;
  as_quat_copy( out, tmp );
end;
  
procedure as_quat_exp( var out: as_quatern; in1: as_quatern );
var
  tmp: as_quatern;
  len, len1: single;
begin
  len := sqrt( sqr( in1.x ) + sqr( in1.y ) + sqr( in1.z ));
  if ( len > 0.0 ) then
    len1 := sin( len ) / len else 
    len1 := 1.0;
  tmp.w := cos( len );
  tmp.x := in1.x * len1;
  tmp.y := in1.y * len1;
  tmp.z := in1.z * len1;
  as_quat_copy( out, tmp );
end;

procedure as_quat_log( var out: as_quatern; in1: as_quatern );
var
  tmp: as_quatern;
  len: single;
begin
  len := sqrt( sqr( in1.x ) + sqr( in1.y ) + sqr( in1.z ));
  if ( in1.w <> 0.0 ) then
    len := arctan( len / in1.w ) else
    len := 3.1415926536 / 2.0;
  tmp.w := 0.0;
  tmp.x := in1.x * len;
  tmp.y := in1.y * len;
  tmp.z := in1.z * len;
  as_quat_copy( out, tmp );
end;

procedure as_quat_lndif( var out: as_quatern; in1, in2: as_quatern );
var
  inv, dif, tmp: as_quatern;
  len, len1, s: single;
begin
  as_quat_inverse( inv, in1 );
  as_quat_mul( dif, inv, in2 );
  len := sqrt( sqr( dif.x ) + sqr( dif.y ) + sqr( dif.z ));
  s := as_quat_dot( in1, in2 );
  if ( s <> 0.0 ) then
    len1 := arctan( len / s ) else
    len1 := 3.1415926536 / 2.0;
  if ( len <> 0 ) then len1 := len1 / len;
  tmp.w := 0;
  tmp.x := dif.x * len1;
  tmp.y := dif.y * len1;
  tmp.z := dif.z * len1;
  as_quat_copy( out, tmp );
end;

procedure as_quat_slerp( var out: as_quatern; in1, in2: as_quatern; spin, alpha: single );
var
  tmp: as_quatern;
  k1, k2, angle, anglespin, sina, cosa: single;
  flip: longint;
begin
  cosa := as_quat_dotunit( in1, in2 );
  if ( cosa < 0.0 ) then
    begin
      cosa := - cosa; 
      flip := -1;
    end else
      flip := 1;
  if (( 1.0-cosa ) < as_epsilon ) then
    begin
      k1 := 1.0 - alpha;
      k2 := alpha;
    end else begin
      angle := arccos( cosa );
      sina := sin( angle );
      anglespin := angle + spin * 3.1415926536;
      k1 := sin( angle - alpha*anglespin ) / sina;
      k2 := sin( alpha*anglespin ) / sina;
    end;
  k2 *= flip;
  tmp.w := k1 * in1.w + k2 * in2.w;
  tmp.x := k1 * in1.x + k2 * in2.x;
  tmp.y := k1 * in1.y + k2 * in2.y;
  tmp.z := k1 * in1.z + k2 * in2.z;
  as_quat_copy( out, tmp );    
end;

procedure as_quat_slerpl( var out: as_quatern; in1, in2: as_quatern; spin, alpha: single );
var
  tmp: as_quatern;
  k1, k2, angle, anglespin, sina, cosa: single;
  flip: longint;
begin
  cosa := as_quat_dotunit( in1, in2 );
  if (( 1.0-cosa ) < as_epsilon ) then
    begin
      k1 := 1.0 - alpha;
      k2 := alpha;
    end else begin
      angle := arccos( cosa );
      sina := sin( angle );
      anglespin := angle;
      k1 := sin( angle - alpha*anglespin ) / sina;
      k2 := sin( alpha*anglespin ) / sina;
    end;
  tmp.w := k1 * in1.w + k2 * in2.w;
  tmp.x := k1 * in1.x + k2 * in2.x;
  tmp.y := k1 * in1.y + k2 * in2.y;
  tmp.z := k1 * in1.z + k2 * in2.z;
  as_quat_copy( out, tmp );    
end;

procedure as_quat_toang( var ang, x, y, z: single; in1: as_quatern );
var
  q: as_quatern;
  s, omega: single;
begin
  as_quat_normalize( q, in1 );
  omega := arccos( q.w );
  ang := 2.0 * omega;
  s := sin( omega );
  if ( abs( s ) > as_epsilon ) then
    begin
      s := 1.0 / s;
      x := q.x * s;
      y := q.y * s;
      z := q.z * s;
    end else begin
      x := 0;
      y := 0;
      z := 0;
    end;
end;

procedure as_quat_fromang( var out: as_quatern; ang, x, y, z: single );
var
  s, omega: single;
begin
  omega := 0.5 * ang;
  s := sin( omega );
  out.w := cos( omega );
  out.x := x * s;
  out.y := y * s;
  out.z := z * s;
end;

procedure as_quat_tomatrix( var out: as_matrix; in1: as_quatern );
var
  x2, y2, z2, wx, wy, wz, xx, xy, xz, yy, yz, zz: single;
begin
  x2 := in1.x + in1.x;
  y2 := in1.y + in1.y;
  z2 := in1.z + in1.z;
  wx := in1.w * x2;
  wy := in1.w * y2;
  wz := in1.w * z2;
  xx := in1.x * x2;
  xy := in1.x * y2;
  xz := in1.x * z2;
  yy := in1.y * y2;
  yz := in1.y * z2;
  zz := in1.z * z2;
  out[ 0, 0 ] := 1.0 - yy - zz;
  out[ 0, 1 ] := xy + wz;
  out[ 0, 2 ] := xz - wy;
  out[ 0, 3 ] := 0.0;
  out[ 1, 0 ] := xy - wz;
  out[ 1, 1 ] := 1.0 - xx - zz;
  out[ 1, 2 ] := yz + wx;
  out[ 1, 3 ] := 0.0;
  out[ 2, 0 ] := xz + wy;
  out[ 2, 1 ] := yz - wx;
  out[ 2, 2 ] := 1.0 - xx - yy;
  out[ 2, 3 ] := 0.0;
end;

procedure as_quat_toinvmatrix( var out: as_matrix; in1: as_quatern );
var
  x2, y2, z2, wx, wy, wz, xx, xy, xz, yy, yz, zz: single;
begin
  x2 := in1.x + in1.x;
  y2 := in1.y + in1.y;
  z2 := in1.z + in1.z;
  wx := in1.w * x2;
  wy := in1.w * y2;
  wz := in1.w * z2;
  xx := in1.x * x2;
  xy := in1.x * y2;
  xz := in1.x * z2;
  yy := in1.y * y2;
  yz := in1.y * z2;
  zz := in1.z * z2;
  out[ 0, 0 ] := 1.0 - yy - zz;
  out[ 0, 1 ] := xy - wz;
  out[ 0, 2 ] := xz + wy;
  out[ 0, 3 ] := 0.0;
  out[ 1, 0 ] := xy + wz;
  out[ 1, 1 ] := 1.0 - xx - zz;
  out[ 1, 2 ] := yz - wx;
  out[ 1, 3 ] := 0.0;
  out[ 2, 0 ] := xz - wy;
  out[ 2, 1 ] := yz + wx;
  out[ 2, 2 ] := 1.0 - xx - yy;
  out[ 2, 3 ] := 0.0;
end;

procedure as_quat_frommatrix( var out: as_quatern; in1: as_matrix );
const
  nxt: array[ 0..2 ] of longint = ( 1, 2, 0 );
var
  tr, s: single;
  q: array[ 0..3 ] of single;
  i, j, k: longint;
begin
  tr := in1[ 0, 0 ] + in1[ 1, 1 ] + in1[ 2, 2 ];
  if ( tr > 0.0 ) then
    begin
      s := sqrt( tr + 1.0 );
      out.w := s / 2.0;
      s := 0.5 / s;
      out.x := ( in1[ 1, 2 ] - in1[ 2, 1 ] ) * s;
      out.y := ( in1[ 2, 0 ] - in1[ 0, 2 ] ) * s;
      out.z := ( in1[ 0, 1 ] - in1[ 1, 0 ] ) * s;
    end else begin
      i := 0;
      if ( in1[ 1, 1 ] > in1[ 0, 0 ] ) then i := 1;
      if ( in1[ 2, 2 ] > in1[ i, i ] ) then i := 2;
      j := nxt[i];
      k := nxt[j];
      s := sqrt(( in1[ i, i ] - ( in1[ j, j ] + in1[ k, k ] )) + 1.0 );
      q[ i ] := s / 2.0;
      if ( s <> 0.0 ) then
        s := 0.5 / s;
      q[ 3 ] := ( in1[ j, k ] - in1[ k, j ] ) * s;
      q[ j ] := ( in1[ i, j ] - in1[ j, i ] ) * s;
      q[ k ] := ( in1[ i, k ] - in1[ k, i ] ) * s;    // [bug]
      out.w := q[3];
      out.x := q[0];
      out.y := q[1];
      out.z := q[2];
    end;
end;

end.