
unit as_matr;

// ==========================================================================
//  autospace3d - version 0.90 / o1.o8.2ooo
//  freepascal (clax) keyframing & (opengl) 3d engine - matrix math.
// --------------------------------------------------------------------------
//  copyright (c)2ooo. remage / fresh!mindworkz.
// --------------------------------------------------------------------------

interface

uses 
  math, as_type, as_vect;

// --------------------------------------------------------------------------

procedure as_matr_zero( var out: as_matrix );
procedure as_matr_identity( var out: as_matrix );

procedure as_matr_copy( var out: as_matrix; in1: as_matrix );

procedure as_matr_print( in1: as_matrix );

procedure as_matr_add( var out: as_matrix; in1, in2: as_matrix );
procedure as_matr_sub( var out: as_matrix; in1, in2: as_matrix );
procedure as_matr_mul( var out: as_matrix; in1, in2: as_matrix );

procedure as_matr_transpose( var out: as_matrix; in1: as_matrix );

function as_matr_inverse( var out: as_matrix; in1: as_matrix ): dword;
function as_matr_invscale( var out: as_matrix; in1: as_matrix ): dword;

procedure as_matr_normalize( var out: as_matrix; in1: as_matrix );

procedure as_matr_toeuler( var out: as_vector; in1: as_matrix );

procedure as_matr_pretrans( var out: as_matrix; in1: as_matrix; in2: as_vector );

procedure as_matr_settrans( var out: as_matrix; in1: as_vector );
procedure as_matr_setscale( var out: as_matrix; in1: as_vector );
procedure as_matr_rotatex( var out: as_matrix; angle: single );
procedure as_matr_rotatey( var out: as_matrix; angle: single );
procedure as_matr_rotatez( var out: as_matrix; angle: single );

procedure as_matr_mulvect( var out: as_vector; in1: as_vector; in2: as_matrix );
procedure as_matr_mulnorm( var out: as_vector; in1: as_vector; in2: as_matrix );

// ==========================================================================

implementation

const
  ident : as_matrix =
    (( 1, 0, 0, 0 ), 
     ( 0, 1, 0, 0 ),
     ( 0, 0, 1, 0 ));

// --- matr_zero ------------------------------------------------------------

procedure as_matr_zero( var out: as_matrix );
var
  i, j: longint;
begin 
  for i := 0 to 2 do
    for j := 0 to 3 do
      out[ i, j ] := 0.0;
end;

// --- matr_identity --------------------------------------------------------

procedure as_matr_identity( var out: as_matrix );
var
  i, j: longint;
begin
  for i := 0 to 2 do
    for j := 0 to 3 do
      out[ i, j ] := ident[ i, j ];
end;

// --- matr_copy ------------------------------------------------------------

procedure as_matr_copy( var out: as_matrix; in1: as_matrix );
var
  i, j: longint;
begin
  for i := 0 to 2 do
    for j := 0 to 3 do
      out[ i, j ] := in1[ i, j ];
end;

// --- matr_print -----------------------------------------------------------

procedure as_matr_print( in1: as_matrix );
begin
  writeln( ' xx: ', in1[ 0, 0 ], ' xy: ', in1[ 0, 1 ], ' xz: ', in1[ 0, 2 ], ' xw: ', in1[ 0, 3 ] ); 
  writeln( ' yx: ', in1[ 1, 0 ], ' yy: ', in1[ 1, 1 ], ' yz: ', in1[ 1, 2 ], ' yw: ', in1[ 1, 3 ] ); 
  writeln( ' zx: ', in1[ 2, 0 ], ' zy: ', in1[ 2, 1 ], ' zz: ', in1[ 2, 2 ], ' zw: ', in1[ 2, 3 ] ); 
end;

// --- matr_add -------------------------------------------------------------

procedure as_matr_add( var out: as_matrix; in1, in2: as_matrix );
var
  tmp: as_matrix;
  i, j: longint;
begin
  for i := 0 to 2 do
    for j := 0 to 3 do
      tmp[ i, j ] := in1[ i, j ] + in2[ i, j ];
  as_matr_copy( out, tmp );
end;

// --- matr_sub -------------------------------------------------------------

procedure as_matr_sub( var out: as_matrix; in1, in2: as_matrix );
var
  tmp: as_matrix;
  i, j: longint;
begin
  for i := 0 to 2 do
    for j := 0 to 3 do
      tmp[ i, j ] := in1[ i, j ] - in2[ i, j ];
  as_matr_copy( out, tmp );
end;

// --- matr_mul -------------------------------------------------------------

procedure as_matr_mul( var out: as_matrix; in1, in2: as_matrix );
var
  tmp: as_matrix;
begin
  tmp[ 0, 0 ] := in1[ 0, 0 ] * in2[ 0, 0 ] + in1[ 0, 1 ] * in2[ 1, 0 ] + in1[ 0, 2 ] * in2[ 2, 0 ];
  tmp[ 0, 1 ] := in1[ 0, 0 ] * in2[ 0, 1 ] + in1[ 0, 1 ] * in2[ 1, 1 ] + in1[ 0, 2 ] * in2[ 2, 1 ];
  tmp[ 0, 2 ] := in1[ 0, 0 ] * in2[ 0, 2 ] + in1[ 0, 1 ] * in2[ 1, 2 ] + in1[ 0, 2 ] * in2[ 2, 2 ];
  tmp[ 0, 3 ] := in1[ 0, 0 ] * in2[ 0, 3 ] + in1[ 0, 1 ] * in2[ 1, 3 ] + in1[ 0, 2 ] * in2[ 2, 3 ] + in1[ 0, 3 ];
  tmp[ 1, 0 ] := in1[ 1, 0 ] * in2[ 0, 0 ] + in1[ 1, 1 ] * in2[ 1, 0 ] + in1[ 1, 2 ] * in2[ 2, 0 ];
  tmp[ 1, 1 ] := in1[ 1, 0 ] * in2[ 0, 1 ] + in1[ 1, 1 ] * in2[ 1, 1 ] + in1[ 1, 2 ] * in2[ 2, 1 ];
  tmp[ 1, 2 ] := in1[ 1, 0 ] * in2[ 0, 2 ] + in1[ 1, 1 ] * in2[ 1, 2 ] + in1[ 1, 2 ] * in2[ 2, 2 ];
  tmp[ 1, 3 ] := in1[ 1, 0 ] * in2[ 0, 3 ] + in1[ 1, 1 ] * in2[ 1, 3 ] + in1[ 1, 2 ] * in2[ 2, 3 ] + in1[ 1, 3 ];
  tmp[ 2, 0 ] := in1[ 2, 0 ] * in2[ 0, 0 ] + in1[ 2, 1 ] * in2[ 1, 0 ] + in1[ 2, 2 ] * in2[ 2, 0 ];
  tmp[ 2, 1 ] := in1[ 2, 0 ] * in2[ 0, 1 ] + in1[ 2, 1 ] * in2[ 1, 1 ] + in1[ 2, 2 ] * in2[ 2, 1 ];
  tmp[ 2, 2 ] := in1[ 2, 0 ] * in2[ 0, 2 ] + in1[ 2, 1 ] * in2[ 1, 2 ] + in1[ 2, 2 ] * in2[ 2, 2 ];
  tmp[ 2, 3 ] := in1[ 2, 0 ] * in2[ 0, 3 ] + in1[ 2, 1 ] * in2[ 1, 3 ] + in1[ 2, 2 ] * in2[ 2, 3 ] + in1[ 2, 3 ];
  as_matr_copy( out, tmp );
end;

// --- matr_transpose -------------------------------------------------------

procedure as_matr_transpose( var out: as_matrix; in1: as_matrix );
var
  tmp: as_matrix;
  i, j: longint;
begin
  for i := 0 to 2 do
    for j := 0 to 2 do
      tmp[ i, j ] := in1[ j, i ];
  for i := 0 to 2 do
    tmp[ i, 3 ] := in1[ i, 3 ];
  as_matr_copy( out, tmp );
end;

// --- matr_inverse ---------------------------------------------------------

function as_matr_inverse( var out: as_matrix; in1: as_matrix ): dword;
var
  tmp: as_matrix;
  scale: single;
  i, j: longint;
begin
  for i := 0 to 2 do
    begin
      scale := sqr( in1[ i, 0 ] ) + sqr( in1[ i, 1 ] ) + sqr( in1[ i, 2 ] );
      if ( scale = 0.0 ) then
        begin
          as_matr_identity( out );
          as_matr_inverse := as_err_singular_matrix;
          exit;
        end;
      scale := 1.0 / scale;
      for j := 0 to 2 do
        tmp [ i, j ] := in1[ j, i ] * scale;
      tmp[ i, 3 ] := - ( tmp[ i, 0 ] * in1[ 0, 3 ] + tmp[ i, 1 ] * in1[ 1, 3 ] + tmp[ i, 2 ] * in1[ 2, 3 ] );
    end;
  as_matr_copy( out, tmp );
  as_matr_inverse := as_err_ok;
end;

// --- matr_invscale --------------------------------------------------------

function as_matr_invscale( var out: as_matrix; in1: as_matrix ): dword;
var
  tmp: as_matrix;
  scale: single;
  i, j: longint;
begin
  for i := 0 to 2 do
    begin
      scale := sqr( in1[ i, 0 ] ) + sqr( in1[ i, 1 ] ) + sqr( in1[ i, 2 ] );
      if ( scale = 0.0 ) then
        begin
          as_matr_identity( out );
          as_matr_invscale := as_err_singular_matrix;
          exit;
        end;
      scale := 1.0 / scale;
      for j := 0 to 2 do
        tmp [ i, j ] := in1[ j, i ] * scale;
      tmp[ i, 3 ] := in1[ i, 3 ];
    end;
  as_matr_copy( out, tmp );
  as_matr_invscale := as_err_ok;
end;

// --- matr_normalize -------------------------------------------------------

procedure as_matr_normalize( var out: as_matrix; in1: as_matrix );
var
  tmp: as_matrix;
  len: single;
  i, j: longint;
begin
  for i := 0 to 2 do
    begin
      len := sqrt( sqr( in1[ i, 0 ] ) + sqr( in1[ i, 1 ] ) + sqr( in1[ i, 2 ] ));
      if ( len <> 0.0 ) then
        len := 1.0 / len else
        len := 1.0;
      for j := 0 to 2 do
        tmp[ i, j ] := in1[ i, j ] * len;
      tmp[ i, 3 ] := in1[ i, 3 ];
    end;
  as_matr_copy( out, tmp );
end;

// --- matr_toeuler ---------------------------------------------------------

procedure as_matr_toeuler( var out: as_vector; in1: as_matrix );
var
  siny, cosy: single;
begin
  siny := in1[ 2, 0 ];
  cosy := sqrt( 1.0 - sqr( siny ));
  if (( in1[ 0, 0 ] < 0.0 ) and ( in1[ 2, 2 ] < 0.0 )) then
    cosy := -cosy;
  if ( cosy <> 0.0 ) then
    begin
      out.x := arctan2( in1[ 2, 1 ] / cosy, in1[ 2, 2 ] / cosy );
      out.y := arctan2( siny, cosy );
      out.z := arctan2( in1[ 1, 0 ] / cosy, in1[ 0, 0 ] / cosy );
    end else begin
      out.x := arctan2( -in1[ 1, 2], in1[ 1, 1 ] );
      out.y := arcsin( siny );
      out.z := 0.0;
    end;
end;

// --- matr_pretrans --------------------------------------------------------

procedure as_matr_pretrans( var out: as_matrix; in1: as_matrix; in2: as_vector );
var
  tmp: as_matrix;
  i, j: longint;
begin
  for i := 0 to 2 do
    for j := 0 to 2 do
      tmp[ i, j ] := in1[ i, j ];
  tmp[ 0, 3 ] := in1[ 0, 0 ] * in2.x + in1[ 0, 1 ] * in2.y + in1[ 0, 2 ] * in2.z + in1[ 0, 3 ];
  tmp[ 1, 3 ] := in1[ 1, 0 ] * in2.x + in1[ 1, 1 ] * in2.y + in1[ 1, 2 ] * in2.z + in1[ 1, 3 ];
  tmp[ 2, 3 ] := in1[ 2, 0 ] * in2.x + in1[ 2, 1 ] * in2.y + in1[ 2, 2 ] * in2.z + in1[ 2, 3 ];
  as_matr_copy( out, tmp );
end;

// --- matr_settrans --------------------------------------------------------

procedure as_matr_settrans( var out: as_matrix; in1: as_vector );
begin
  as_matr_identity( out );
  out[ 0, 3 ] := in1.x;
  out[ 1, 3 ] := in1.y;
  out[ 2, 3 ] := in1.z;
end;

// --- matr_setscale --------------------------------------------------------

procedure as_matr_setscale( var out: as_matrix; in1: as_vector );
begin
  as_matr_identity( out );
  out[ 0, 0 ] := in1.x;
  out[ 1, 1 ] := in1.y;
  out[ 2, 2 ] := in1.z;
end;

// --- matr_rotatex ---------------------------------------------------------

procedure as_matr_rotatex( var out: as_matrix; angle: single );
var
  sinx, cosx: single;
begin
  sinx := sin( angle );
  cosx := cos( angle );
  as_matr_identity( out );
  out[ 1, 1 ] :=  cosx;  
  out[ 1, 2 ] :=  sinx;  
  out[ 2, 1 ] := -sinx;  
  out[ 2, 2 ] :=  cosx;  
end;

// --- matr_rotatey ---------------------------------------------------------

procedure as_matr_rotatey( var out: as_matrix; angle: single );
var
  siny, cosy: single;
begin
  siny := sin( angle );
  cosy := cos( angle );
  as_matr_identity( out );
  out[ 0, 0 ] :=  cosy;  
  out[ 0, 2 ] := -siny;  
  out[ 2, 0 ] :=  siny;  
  out[ 2, 2 ] :=  cosy;  
end;

// --- matr_rotatez ---------------------------------------------------------

procedure as_matr_rotatez( var out: as_matrix; angle: single );
var
  sinz, cosz: single;
begin
  sinz := sin( angle );
  cosz := cos( angle );
  as_matr_identity( out );
  out[ 0, 0 ] :=  cosz;  
  out[ 0, 1 ] :=  sinz;  
  out[ 1, 0 ] := -sinz;  
  out[ 1, 1 ] :=  cosz;  
end;

// --- matr_mulvect ---------------------------------------------------------

procedure as_matr_mulvect( var out: as_vector; in1: as_vector; in2: as_matrix );
var
  tmp: as_vector;
begin
  tmp.x := in1.x * in2[ 0, 0 ] + in1.y * in2[ 0, 1 ] + in1.z * in2[ 0, 2 ] + in2[ 0, 3 ];
  tmp.y := in1.x * in2[ 1, 0 ] + in1.y * in2[ 1, 1 ] + in1.z * in2[ 1, 2 ] + in2[ 1, 3 ];
  tmp.z := in1.x * in2[ 2, 0 ] + in1.y * in2[ 2, 1 ] + in1.z * in2[ 2, 2 ] + in2[ 2, 3 ];
  as_vect_copy( out, tmp );
end;

// --- matr_mulnorm ---------------------------------------------------------

procedure as_matr_mulnorm( var out: as_vector; in1: as_vector; in2: as_matrix );
var
  tmp: as_vector;
begin
  tmp.x := in1.x * in2[ 0, 0 ] + in1.y * in2[ 0, 1 ] + in1.z * in2[ 0, 2 ];
  tmp.y := in1.x * in2[ 1, 0 ] + in1.y * in2[ 1, 1 ] + in1.z * in2[ 1, 2 ];
  tmp.z := in1.x * in2[ 2, 0 ] + in1.y * in2[ 2, 1 ] + in1.z * in2[ 2, 2 ];
  as_vect_copy( out, tmp );
end;

end.