
unit as_3ds;

// ==========================================================================
//  autospace3d - version 0.90 / o1.o8.2ooo
//  freepascal (clax) keyframing & (opengl) 3d engine - 3ds 3.0+ driver.
// --------------------------------------------------------------------------
//  copyright (c)2ooo. remage / fresh!mindworkz.
// --------------------------------------------------------------------------

{x$define as_3ds_debug}
{$define as_3ds_swapyz}

// ==========================================================================

interface

uses
  math, as_type, as_vect, as_matr, as_quat, as_splin, autosp3d, zfs;

function as3d_loadmesh3ds( var f: file ): longint;
function as3d_loadmotion3ds( var f: file ): longint;

// ==========================================================================

implementation

const
  chunk_rgbf         = $0010;
  chunk_rgbb         = $0011;
  chunk_prj          = $C23D;
  chunk_mli          = $3DAA;
  chunk_main         = $4D4D;
  chunk_objmesh      = $3D3D;
  chunk_bkgcolor     = $1200;
  chunk_ambcolor     = $2100;
  chunk_objblock     = $4000;
  chunk_trimesh      = $4100;
  chunk_vertlist     = $4110;
  chunk_vertflags    = $4111;
  chunk_facelist     = $4120;
  chunk_facemat      = $4130;
  chunk_maplist      = $4140;
  chunk_smoolist     = $4150;
  chunk_trmatrix     = $4160;
  chunk_meshcolor    = $4165;
  chunk_txtinfo      = $4170;
  chunk_light        = $4600;
  chunk_spotlight    = $4610;
  chunk_camera       = $4700;
  chunk_hierarchy    = $4F00;
  chunk_viewport     = $7001;
  chunk_material     = $AFFF;
  chunk_matname      = $A000;
  chunk_ambient      = $A010;
  chunk_diffuse      = $A020;
  chunk_specular     = $A030;
  chunk_texture      = $A200;
  chunk_bumpmap      = $A230;
  chunk_reflection   = $A220;
  chunk_mapfile      = $A300;
  chunk_mapflags     = $A351;
  chunk_mapuscale    = $A354;
  chunk_mapvscale    = $A356;
  chunk_mapuoffset   = $A358;
  chunk_mapvoffset   = $A35A;
  chunk_keyframer    = $B000;
  chunk_ambientkey   = $B001;
  chunk_trackinfo    = $B002;
  chunk_trackobjname = $B010;
  chunk_trackpivot   = $B013;
  chunk_trackpos     = $B020;
  chunk_trackrotate  = $B021;
  chunk_trackscale   = $B022;
  chunk_trackmorph   = $B026;
  chunk_trackhide    = $B029;
  chunk_objnumber    = $B030;
  chunk_trackcamera  = $B003;
  chunk_trackfov     = $B023;
  chunk_trackroll    = $B024;
  chunk_trackcamtgt  = $B004;
  chunk_tracklight   = $B005;
  chunk_trackligtgt  = $B006;
  chunk_trackspotl   = $B007;
  chunk_trackcolor   = $B025;
  chunk_frames       = $B008;
  chunk_dummyname    = $B011;
  chunk_maprotangle  = $A35C;
  chunk_shininess    = $A040;
  chunk_shinstrength = $A041;
  chunk_transparency = $A050;
  chunk_transfalloff = $A052;
  chunk_refblur      = $A053;
  chunk_selfillum    = $A084;
  chunk_twosided     = $A081;
  chunk_transadd     = $A083;
  chunk_wireon       = $A085;
  chunk_soften       = $A08C;
  chunk_mattype      = $A100;
  chunk_amountof     = $0030;

type
  pchunk = ^chunk;
  chunk = record
    id: word;
    size: longint;
    end;

  reader = record
    id: word;
    sub: longint;
    func: function( var f: file ): longint;
    end;

function read_null        ( var f: file ): longint; forward;       // (skip chunk)
function read_rgbf        ( var f: file ): longint; forward;       // rgb float
function read_rgbb        ( var f: file ): longint; forward;       // rgb byte
function read_amountof    ( var f: file ): longint; forward;       // amount of
function read_asciiz      ( var f: file ): longint; forward;       // asciiz string
function read_trimesh     ( var f: file ): longint; forward;       // triangular mesh
function read_vertlist    ( var f: file ): longint; forward;       // vertex list
function read_facelist    ( var f: file ): longint; forward;       // face list
function read_facemat     ( var f: file ): longint; forward;       // face material
function read_maplist     ( var f: file ): longint; forward;       // mapping list
function read_trmatrix    ( var f: file ): longint; forward;       // transformation matrix
function read_light       ( var f: file ): longint; forward;       // light
function read_spotlight   ( var f: file ): longint; forward;       // spotlight
function read_camera      ( var f: file ): longint; forward;       // camera
function read_material    ( var f: file ): longint; forward;       // material
function read_matname     ( var f: file ): longint; forward;       // material name
function read_frames      ( var f: file ): longint; forward;       // number of frames
function read_objnumber   ( var f: file ): longint; forward;       // object number
function read_trackobjname( var f: file ): longint; forward;       // track object name
function read_dummyname   ( var f: file ): longint; forward;       // dummy object name
function read_trackpivot  ( var f: file ): longint; forward;       // track pivot point
function read_trackpos    ( var f: file ): longint; forward;       // track position
function read_trackcolor  ( var f: file ): longint; forward;       // track color
function read_trackrot    ( var f: file ): longint; forward;       // track rotation
function read_trackscale  ( var f: file ): longint; forward;       // track scale
function read_trackfov    ( var f: file ): longint; forward;       // track fov
function read_trackroll   ( var f: file ): longint; forward;       // track roll
function read_trackmorph  ( var f: file ): longint; forward;       // track morph
function read_trackhide   ( var f: file ): longint; forward;       // track hide
function read_mattype     ( var f: file ): longint; forward;       // material: type
function read_mattwosided ( var f: file ): longint; forward;       // material: two sided
function read_matsoften   ( var f: file ): longint; forward;       // material: soften
function read_matwire     ( var f: file ): longint; forward;       // material: wire
function read_mattransadd ( var f: file ): longint; forward;       // material: transparency
function read_mapflags    ( var f: file ): longint; forward;       // map flags
function read_mapfile     ( var f: file ): longint; forward;       // map file
function read_mapuscale   ( var f: file ): longint; forward;       // map 1/u scale
function read_mapvscale   ( var f: file ): longint; forward;       // map 1/v scale
function read_mapuoffset  ( var f: file ): longint; forward;       // map u offset
function read_mapvoffset  ( var f: file ): longint; forward;       // map v offset
function read_maprotangle ( var f: file ): longint; forward;       // map rotation angle

const

  chunk_world : array[ 0..50 ] of reader =
   (( id:chunk_rgbf;         sub:0; func:read_rgbf ),
    ( id:chunk_rgbb;         sub:0; func:read_rgbb ),
    ( id:chunk_amountof;     sub:0; func:read_amountof ),
    ( id:chunk_prj;          sub:1; func:read_null ),
    ( id:chunk_mli;          sub:1; func:read_null ),
    ( id:chunk_main;         sub:1; func:read_null ),
    ( id:chunk_objmesh;      sub:1; func:read_null ),
    ( id:chunk_bkgcolor;     sub:1; func:read_null ),
    ( id:chunk_ambcolor;     sub:1; func:read_null ),
    ( id:chunk_objblock;     sub:1; func:read_asciiz ),
    ( id:chunk_trimesh;      sub:1; func:read_trimesh ),
    ( id:chunk_vertlist;     sub:0; func:read_vertlist ),
    ( id:chunk_vertflags;    sub:0; func:read_null ),
    ( id:chunk_facelist;     sub:1; func:read_facelist ),
    ( id:chunk_meshcolor;    sub:0; func:read_null ),
    ( id:chunk_facemat;      sub:0; func:read_facemat ),
    ( id:chunk_maplist;      sub:0; func:read_maplist ),
    ( id:chunk_txtinfo;      sub:0; func:read_null ),
    ( id:chunk_smoolist;     sub:0; func:read_null ),
    ( id:chunk_trmatrix;     sub:0; func:read_trmatrix ),
    ( id:chunk_light;        sub:1; func:read_light ),
    ( id:chunk_spotlight;    sub:0; func:read_spotlight ),
    ( id:chunk_camera;       sub:0; func:read_camera ),
    ( id:chunk_hierarchy;    sub:1; func:read_null ),
    ( id:chunk_viewport;     sub:0; func:read_null ),
    ( id:chunk_material;     sub:1; func:read_material ),
    ( id:chunk_matname;      sub:0; func:read_matname ),
    ( id:chunk_ambient;      sub:1; func:read_null ),
    ( id:chunk_diffuse;      sub:1; func:read_null ),
    ( id:chunk_specular;     sub:1; func:read_null ),
    ( id:chunk_texture;      sub:1; func:read_null ),
    ( id:chunk_bumpmap;      sub:1; func:read_null ),
    ( id:chunk_reflection;   sub:1; func:read_null ),
    ( id:chunk_mapfile;      sub:0; func:read_mapfile ),
    ( id:chunk_mapflags;     sub:0; func:read_mapflags ),
    ( id:chunk_mapuscale;    sub:0; func:read_mapuscale ),
    ( id:chunk_mapvscale;    sub:0; func:read_mapvscale ),
    ( id:chunk_mapuoffset;   sub:0; func:read_mapuoffset ),
    ( id:chunk_mapvoffset;   sub:0; func:read_mapvoffset ),
    ( id:chunk_maprotangle;  sub:0; func:read_maprotangle ),
    ( id:chunk_shininess;    sub:1; func:read_null ),
    ( id:chunk_shinstrength; sub:1; func:read_null ),
    ( id:chunk_transparency; sub:1; func:read_null ),
    ( id:chunk_transfalloff; sub:1; func:read_null ),
    ( id:chunk_refblur;      sub:1; func:read_null ),
    ( id:chunk_selfillum;    sub:1; func:read_null ),
    ( id:chunk_twosided;     sub:0; func:read_mattwosided ),
    ( id:chunk_transadd;     sub:0; func:read_mattransadd ),
    ( id:chunk_wireon;       sub:0; func:read_matwire ),
    ( id:chunk_soften;       sub:0; func:read_matsoften ),
    ( id:chunk_mattype;      sub:0; func:read_mattype ));

  chunk_keyfr : array[ 0..21 ] of reader =
   (( id:chunk_main;         sub:1; func:read_null ),
    ( id:chunk_keyframer;    sub:1; func:read_null ),
    ( id:chunk_ambientkey;   sub:1; func:read_null ),
    ( id:chunk_trackinfo;    sub:1; func:read_null ),
    ( id:chunk_frames;       sub:0; func:read_frames ),
    ( id:chunk_trackobjname; sub:0; func:read_trackobjname ),
    ( id:chunk_dummyname;    sub:0; func:read_dummyname ),
    ( id:chunk_trackpivot;   sub:0; func:read_trackpivot ),
    ( id:chunk_trackpos;     sub:0; func:read_trackpos ),
    ( id:chunk_trackcolor;   sub:0; func:read_trackcolor ),
    ( id:chunk_trackrotate;  sub:0; func:read_trackrot ),
    ( id:chunk_trackscale;   sub:0; func:read_trackscale ),
    ( id:chunk_trackmorph;   sub:0; func:read_trackmorph ),
    ( id:chunk_trackhide;    sub:0; func:read_trackhide ),
    ( id:chunk_objnumber;    sub:0; func:read_objnumber ),
    ( id:chunk_trackcamera;  sub:1; func:read_null ),
    ( id:chunk_trackcamtgt;  sub:1; func:read_null ),
    ( id:chunk_tracklight;   sub:1; func:read_null ),
    ( id:chunk_trackligtgt;  sub:1; func:read_null ),
    ( id:chunk_trackspotl;   sub:1; func:read_null ),
    ( id:chunk_trackfov;     sub:0; func:read_trackfov ),
    ( id:chunk_trackroll;    sub:0; func:read_trackroll ));

var
  c_chunk_last, c_chunk_prev, c_chunk_curr, c_id: longint;
  c_string: string[64];
  c_node: pointer;


const
  dbstr: string = '';
var
  debug: text;

function hex( num: longint; siz: byte ): string;
const
  nibble: string='0123456789ABCDEF';
var
  temp: string;
  i: byte;
begin
  temp := '$';
  for i := siz-1 downto 0 do
    begin
      temp := temp + nibble[ ((num shr (i*4)) and 15) +1 ];
    end;
  hex := temp;
end;

// --------------------------------------------------------------------------

procedure as_vect_swap( var out: as_vector );
var
  tmp: single;
begin
{$ifdef as_3ds_swapyz}
  tmp := out.y;
  out.y := out.z;
  out.z := tmp;
{$endif}
end;

// --------------------------------------------------------------------------

procedure as_quat_swap( var out: as_quatern );
var
  tmp: single;
begin
{$ifdef as_3ds_swapyz}
  tmp := out.y;
  out.y := out.z;
  out.z := tmp;
{$endif}
end;

// --------------------------------------------------------------------------

procedure as_matr_swap( var out: as_matrix );
var
  tmp: single;
  i: longint;
begin
{$ifdef as_3ds_swapyz}
  for i := 0 to 2 do
    begin
      tmp := out[ i, 1 ];
      out[ i, 1 ] := out[ i, 2 ];
      out[ i, 2 ] := tmp;
    end;
  for i := 0 to 3 do
    begin
      tmp := out[ 1, i ];
      out[ 1, i ] := out[ 2, i ];
      out[ 2, i ] := tmp;
    end;
{$endif}
end;

// --------------------------------------------------------------------------

procedure clear_map( var map: as_map );
begin
  map.fname := '';
  map.flags := 0;
  map.uscl := 0.0;
  map.vscl := 0.0;
  map.uofs := 0.0;
  map.vofs := 0.0;
  map.angle := 0.0;
end;

// --------------------------------------------------------------------------

procedure clear_mat( var mat: as_material );
begin
  mat.shading := 0;
  mat.flags := 0;
  clear_map( mat.texture );
  clear_map( mat.bump );
  clear_map( mat.reflection );
end;

// --------------------------------------------------------------------------

function alloc_track: as_ptrack;
var
  track : as_ptrack;
begin
  new( track );
  if ( track <> NIL ) then
    begin
      track^.keys := NIL;
      track^.last := NIL;
      track^.flags := 0;
      track^.frames := 0.0;
      track^.numkeys := 0;
    end;
  alloc_track := track;
end;

// --------------------------------------------------------------------------

function add_key( track: as_ptrack; key: as_pkey; frame: longint ): longint;
begin
  if (( track = NIL ) or ( key = NIL )) then
    begin
      add_key := as_err_null_pointer;
      exit;
    end;
  key^.frame := frame;
  key^.next := NIL;
  if ( track^.keys = NIL ) then
    begin
      key^.prev := NIL;
      track^.keys := key;
    end else begin
      key^.prev := track^.last;
      track^.last^.next := key;
    end;
  track^.frames := key^.frame;
  track^.last := key;
  inc( track^.numkeys );
  add_key := as_err_ok;
end;

// ==========================================================================

function read_null( var f: file ): longint;
begin
  read_null := as_err_ok;
end;

// --------------------------------------------------------------------------

function read_rgbf( var f: file ): longint;
const
  rgb: as_pcolor = NIL;
var
  c: array[ 0..2 ] of single;
begin
  case ( c_chunk_last ) of
    chunk_light:
      rgb := @( as_plight( c_node )^.color );
    chunk_ambient:
      rgb := @( as_pmaterial( c_node )^.ambient );
    chunk_diffuse:
      rgb := @( as_pmaterial( c_node )^.diffuse );
    chunk_specular:
      rgb := @( as_pmaterial( c_node )^.specular );
    end;
  blockread( f, c, sizeof( c ));
  if ( rgb <> NIL ) then
    begin
      rgb^.r := c[0];
      rgb^.g := c[1];
      rgb^.b := c[2];
    end;
  read_rgbf := as_err_ok;
end;

// --------------------------------------------------------------------------

function read_rgbb( var f: file ): longint;
const
  rgb: as_pcolor = NIL;
var
  b: array[ 0..2 ] of byte;
begin
  case ( c_chunk_last ) of
    chunk_light:
      rgb := @( as_plight( c_node )^.color );
    chunk_ambient:
      rgb := @( as_pmaterial( c_node )^.ambient );
    chunk_diffuse:
      rgb := @( as_pmaterial( c_node )^.diffuse );
    chunk_specular:
      rgb := @( as_pmaterial( c_node )^.specular );
    end;
  blockread( f, b, sizeof( b ));
  if ( rgb <> NIL ) then
    begin
      rgb^.r := b[0] / 255.0;
      rgb^.g := b[1] / 255.0;
      rgb^.b := b[2] / 255.0;
    end;
  read_rgbb := as_err_ok;
end;

// --------------------------------------------------------------------------

function read_amountof( var f: file ): longint;
const
  fl : psingle = NIL;
var
  w: word;
begin
  case ( c_chunk_last ) of
    chunk_shininess:
      fl := @( as_pmaterial( c_node )^.shininess );
    chunk_shinstrength:
      fl := @( as_pmaterial( c_node )^.shin_strength );
    chunk_transparency:
      fl := @( as_pmaterial( c_node )^.transparency );
    chunk_transfalloff:
      fl := @( as_pmaterial( c_node )^.trans_falloff );
    chunk_refblur:
      fl := @( as_pmaterial( c_node )^.refblur );
    chunk_selfillum:
      fl := @( as_pmaterial( c_node )^.self_illum );
    end;
  blockread( f, w, sizeof( w ));
  if ( fl <> NIL ) then
    begin
      fl^ := w / 100.0;
    end;
  read_amountof := as_err_ok;
end;

// --------------------------------------------------------------------------

function read_asciiz( var f: file ): longint;
var
  c: char;
  i: longint;
begin
  i := 0;
  repeat
    blockread( f, c, 1 );
    if (( c <> #0 ) and ( i < 63 )) then
      begin
        inc( i );
        c_string[i] := c;
      end;
  until eof( f ) or ( c = #0 );
  if ( eof( f )) then
    begin
      read_asciiz := as_err_invalid_file;
      exit;
    end;
  c_string[0] := chr( i );
  read_asciiz := as_err_ok;
end;

// --- read_trimesh ---------------------------------------------------------

function read_trimesh( var f: file ): longint;
var
  obj: as_pobject;
begin
  new( obj );
  obj^.fname := c_string;
  obj^.id := c_id;
  inc( c_id );
  obj^.parent := -1;
  obj^.flags := 0;
  as_vect_zero( obj^.pivot );
  as_vect_zero( obj^.translate );
  as_vect_zero( obj^.scale );
  as_quat_zero( obj^.rotate );
  as_matr_zero( obj^.matrix );
  c_node := obj;
  as3d_addworld( as_obj_object, obj );
  read_trimesh := as_err_ok;
end;

// --- read_vertlist --------------------------------------------------------

function read_vertlist( var f: file ): longint;
var
  obj: as_pobject;
  vert: as_pvertex;
  c: array[ 0..2 ] of single;
  nv: word;
  i, res: longint;
begin
  obj := as_pobject( c_node );
  blockread( f, nv, sizeof( nv ), res );
  if ( res <> sizeof( nv )) then
    begin
      read_vertlist := as_err_invalid_file;
      exit;
    end;
  getmem( vert, nv * sizeof( as_vertex ));
  if ( vert = nil ) then
    begin
      read_vertlist := as_err_out_of_memory;
      exit;
    end;
  obj^.verts := vert;
  obj^.numverts := nv;
  for i := 0 to nv-1 do
    begin
      blockread( f, c, sizeof( c ), res );
      if ( res <> sizeof( c )) then
        begin
          read_vertlist := as_err_invalid_file;
          exit;
        end;
      as_vect_make( vert[i].vert, c[0], c[1], c[2] );
      as_vect_swap( vert[i].vert );
      vert[i].u := 0.0;
      vert[i].v := 0.0;
    end;
  read_vertlist := as_err_ok;
end;

// --- read_facelist --------------------------------------------------------

function read_facelist( var f: file ): longint;
var
  obj: as_pobject;
  face: as_pface;
  c: array[ 0..2 ] of word;
  nf, flags: word;
  i, res: longint;
begin
  obj := as_pobject( c_node );
  blockread( f, nf, sizeof( nf ), res );
  if ( res <> sizeof( nf )) then
    begin
      read_facelist := as_err_invalid_file;
      exit;
    end;
  getmem( face, nf * sizeof( as_face ));
  if ( face = NIL ) then
    begin
      read_facelist := as_err_out_of_memory;
      exit;
    end;
  obj^.faces := face;
  obj^.numfaces := nf;
  for i := 0 to nf-1 do
    begin
      blockread( f, c, sizeof( c ), res );
      if ( res <> sizeof( c )) then
        begin
          read_facelist := as_err_invalid_file;
          exit;
        end;
      blockread( f, flags, sizeof( flags ), res );
      if ( res <> sizeof( flags )) then
        begin
          read_facelist := as_err_invalid_file;
          exit;
        end;
      face[i].a := c[0];
      face[i].b := c[1];
      face[i].c := c[2];
      face[i].pa := @(obj^.verts[c[0]]);
      face[i].pb := @(obj^.verts[c[1]]);
      face[i].pc := @(obj^.verts[c[2]]);
      face[i].flags := flags and $18;
      face[i].mat := 0;
    end;
  read_facelist := as_err_ok;
end;

// --- read_facemat ---------------------------------------------------------

function read_facemat( var f: file ): longint;
var
  face: as_pface;
  node: as_pwnode;
  mat: as_pmaterial;
  n, nf: word;
  i, res: longint;
begin
  face := as_pobject( c_node )^.faces;
  if ( read_asciiz( f ) <> as_err_ok )  then
    begin
      read_facemat := as_err_invalid_file;
      exit;
    end;
  blockread( f, n, sizeof( n ), res );
  if ( res <> sizeof( n )) then
    begin
      read_facemat := as_err_invalid_file;
      exit;
    end;
  as3d_byname( node, c_string );
  if ( node = nil ) then
    begin
      read_facemat := as_err_invalid_object;
      exit;
    end;
  mat := as_pmaterial( node^.obj );
  for i := 0 to n-1 do
    begin
      blockread( f, nf, sizeof( nf ), res );
      if ( res <> sizeof( nf )) then
        begin
          read_facemat := as_err_invalid_file;
          exit;
        end;
      face[nf].mat := mat^.id;
      face[nf].pmat := mat;
    end;
  read_facemat := as_err_ok;
end;

// --- read_maplist ---------------------------------------------------------

function read_maplist( var f: file ): longint;
var
  v: as_pvertex;
  nv: word;
  c: array[ 0..1 ] of single;
  i, res: longint;
begin
  v := as_pobject( c_node )^.verts;
  blockread( f, nv, sizeof( nv ), res );
  if ( res <> sizeof( nv )) then
    begin
      read_maplist := as_err_invalid_file;
      exit;
    end;
  for i := 0 to nv-1 do
    begin
      blockread( f, c, sizeof( c ), res );
      if ( res <> sizeof( c )) then
        begin
          read_maplist := as_err_invalid_file;
          exit;
        end;
      v[i].u := c[0];
      v[i].v := c[1];
    end;
  read_maplist := as_err_ok;
end;

// --- read_trmatrix --------------------------------------------------------

function read_trmatrix( var f: file ): longint;
var
  obj: as_pobject;
  v: as_pvertex;
  piv: as_vector;
  mat: as_matrix;
  pivot: array[ 0..2 ] of single;
  i, j, res: longint;
begin
  obj := as_pobject( c_node );
  v := obj^.verts;
  as_matr_identity( mat );
  for i := 0 to 2 do
    for j := 0 to 2 do
      begin
        blockread( f, mat[ i, j ], sizeof( single ), res );
        if ( res <> sizeof( single )) then
          begin
            read_trmatrix := as_err_invalid_file;
            exit;
          end;
      end;
  blockread( f, pivot, sizeof( pivot ), res );
  if ( res <> sizeof( pivot )) then
    begin
      read_trmatrix := as_err_invalid_file;
      exit;
    end;
{
  as_vect_make( piv, pivot[0], pivot[1], pivot[2] );
  as_vect_swap( piv );
  as_matr_swap( mat );
  as_matr_invscale( mat, mat );
  for i := 0 to obj^.numverts-1 do
    begin
      as_vect_sub( v[i].vert, v[i].vert, piv );
      as_matr_mulvect( v[i].vert, v[i].vert, mat );
    end;
}
  read_trmatrix := as_err_ok;
end;

// --- read_light -----------------------------------------------------------

function read_light( var f: file ): longint;
var
  lit: as_plight;
  c: array[ 0..2 ] of single;
  res: longint;
begin
  new( lit );
  if ( lit = nil ) then
    begin
      read_light := as_err_out_of_memory;
      exit;
    end;
  blockread( f, c, sizeof( c ), res );
  if ( res <> sizeof( c )) then
    begin
      read_light := as_err_invalid_file;
      exit;
    end;
  lit^.fname := c_string;
  lit^.id := c_id;
  inc( c_id );
  c_node := lit;
  lit^.flags := as_lit_omni;
  as_vect_make( lit^.position, c[0], c[1], c[2] );
  as_vect_swap( lit^.position );
  as3d_addworld( as_obj_light, lit );
  read_light := as_err_ok;
end;

// --- read_spotlight -------------------------------------------------------

function read_spotlight( var f: file ): longint;
var
  lit: as_plight;
  c: array[ 0..4 ] of single;
  res: longint;
begin
  lit := as_plight( c_node );
  blockread( f, c, sizeof( c ), res );
  if ( res <> sizeof( c )) then
    begin
      read_spotlight := as_err_invalid_file;
      exit;
    end;
  as_vect_make( lit^.target, c[0], c[1], c[2] );
  as_vect_swap( lit^.target );
  lit^.hotspot := c[3];
  lit^.falloff := c[4];
  lit^.flags := as_lit_spot;
  lit^.roll := 0.0;
  read_spotlight := as_err_ok;
end;

// --- read_camera ----------------------------------------------------------

function read_camera( var f: file ): longint;
var
  cam: as_pcamera;
  c: array[ 0..7 ] of single;
  res: longint;
begin
  new( cam );
  if ( cam = nil ) then
    begin
      read_camera := as_err_out_of_memory;
      exit;
    end;
  blockread( f, c, sizeof( c ), res );
  if ( res <> sizeof( c )) then
    begin
      read_camera := as_err_invalid_file;
      exit;
    end;
  cam^.fname := c_string;
  cam^.id := c_id;
  inc( c_id );
  c_node := cam;
  cam^.roll := c[6];
  cam^.fov := 15.0 / c[7] * 160;
  as_vect_make( cam^.position, c[0], c[1], c[2] );
  as_vect_swap( cam^.position );
  as_vect_make( cam^.target, c[3], c[4], c[5] );
  as_vect_swap( cam^.target );
  as3d_addworld( as_obj_camera, cam );
  read_camera := as_err_ok;
end;

// --- read_ ----------------------------------------------------------------

function read_material( var f: file ): longint;
var
  mat: as_pmaterial;
begin
  new( mat );
  if ( mat = nil ) then
    begin
      read_material := as_err_out_of_memory;
      exit;
    end;
  clear_mat( mat^ );
  mat^.id := c_id;
  inc( c_id );
  c_node := mat;
  as3d_addworld( as_obj_material, mat );
  read_material := as_err_ok;
end;

// --- read_ ----------------------------------------------------------------

function read_matname( var f: file ): longint;
var
  mat: as_pmaterial;
begin
  mat := as_pmaterial( c_node );
  if ( read_asciiz( f ) <> as_err_ok ) then
    begin
      read_matname := as_err_invalid_file;
      exit;
    end;
  mat^.fname := c_string;
  read_matname := as_err_ok;
end;

// --- read_ ----------------------------------------------------------------

function read_mattype( var f: file ): longint;
var
  mat: as_pmaterial;
  typ: word;
  res: longint;
begin
  mat := as_pmaterial( c_node );
  blockread( f, typ, sizeof( typ ), res );
  if ( res <> sizeof( typ )) then
    begin
      read_mattype := as_err_invalid_file;
      exit;
    end;
  mat^.shading := typ;
  read_mattype := as_err_ok;
end;

// --- read_ ----------------------------------------------------------------

function read_mattwosided( var f: file ): longint;
var
  mat: as_pmaterial;
begin
  mat := as_pmaterial( c_node );
  mat^.flags := mat^.flags or as_mat_twosided;
  read_mattwosided := as_err_ok;
end;

// --- read_ ----------------------------------------------------------------

function read_matsoften( var f: file ): longint;
var
  mat: as_pmaterial;
begin
  mat := as_pmaterial( c_node );
  mat^.flags := mat^.flags or as_mat_soften;
  read_matsoften := as_err_ok;
end;

// --- read_ ----------------------------------------------------------------

function read_matwire( var f: file ): longint;
var
  mat: as_pmaterial;
begin
  mat := as_pmaterial( c_node );
  mat^.flags := mat^.flags or as_mat_wireframe;
  read_matwire := as_err_ok;
end;

// --- read_ ----------------------------------------------------------------

function read_mattransadd( var f: file ): longint;
var
  mat: as_pmaterial;
begin
  mat := as_pmaterial( c_node );
  mat^.flags := mat^.flags or as_mat_transparent;
  read_mattransadd := as_err_ok;
end;

// --- read_ ----------------------------------------------------------------

function read_mapfile( var f: file ): longint;
var
  mat: as_pmaterial;
  map: as_pmap;
begin
  mat := as_pmaterial( c_node );
  if ( read_asciiz( f ) <> as_err_ok ) then
    begin
      read_mapfile := as_err_invalid_file;
      exit;
    end;
  case ( c_chunk_last ) of
    chunk_texture:
      map := addr( mat^.texture );
    chunk_bumpmap:
      map := addr( mat^.bump );
    chunk_reflection:
      map := addr( mat^.reflection );
    end;
  map^.fname := c_string;
  read_mapfile := as_err_ok;
end;

// --- read_ ----------------------------------------------------------------

function read_mapflags( var f: file ): longint;
var
  mat: as_pmaterial;
  map: as_pmap;
  flags: word;
  res: longint;
begin
  mat := as_pmaterial( c_node );
  blockread( f, flags, sizeof( flags ), res );
  if ( res <> sizeof( flags )) then
    begin
      read_mapflags := as_err_invalid_file;
      exit;
    end;
  map := nil;
  case ( c_chunk_last ) of
    chunk_texture:
      map := addr( mat^.texture );
    chunk_bumpmap:
      map := addr( mat^.bump );
    chunk_reflection:
      map := addr( mat^.reflection );
    end;
  if ( map <> nil ) then
    map^.flags := flags;
  read_mapflags := as_err_ok;
end;

// --- read_ ----------------------------------------------------------------

function read_mapuscale( var f: file ): longint;
var
  mat: as_pmaterial;
  map: as_pmap;
  u: single;
  res: longint;
begin
  mat := as_pmaterial( c_node );
  blockread( f, u, sizeof( u ), res );
  if ( res <> sizeof( u )) then
    begin
      read_mapuscale := as_err_invalid_file;
      exit;
    end;
  map := nil;
  case ( c_chunk_last ) of
    chunk_texture:
      map := addr( mat^.texture );
    chunk_bumpmap:
      map := addr( mat^.bump );
    chunk_reflection:
      map := addr( mat^.reflection );
    end;
  if ( map <> nil ) then
    map^.uscl := u;
  read_mapuscale := as_err_ok;
end;

// --- read_ ----------------------------------------------------------------

function read_mapvscale( var f: file ): longint;
var
  mat: as_pmaterial;
  map: as_pmap;
  v: single;
  res: longint;
begin
  mat := as_pmaterial( c_node );
  blockread( f, v, sizeof( v ), res );
  if ( res <> sizeof( v )) then
    begin
      read_mapvscale := as_err_invalid_file;
      exit;
    end;
  map := nil;
  case ( c_chunk_last ) of
    chunk_texture:
      map := addr( mat^.texture );
    chunk_bumpmap:
      map := addr( mat^.bump );
    chunk_reflection:
      map := addr( mat^.reflection );
    end;
  if ( map <> nil ) then
    map^.vscl := v;
  read_mapvscale := as_err_ok;
end;

// --- read_ ----------------------------------------------------------------

function read_mapuoffset( var f: file ): longint;
var
  mat: as_pmaterial;
  map: as_pmap;
  u: single;
  res: longint;
begin
  mat := as_pmaterial( c_node );
  blockread( f, u, sizeof( u ), res );
  if ( res <> sizeof( u )) then
    begin
      read_mapuoffset := as_err_invalid_file;
      exit;
    end;
  map := nil;
  case ( c_chunk_last ) of
    chunk_texture:
      map := addr( mat^.texture );
    chunk_bumpmap:
      map := addr( mat^.bump );
    chunk_reflection:
      map := addr( mat^.reflection );
    end;
  if ( map <> nil ) then
    map^.uofs := u;
  read_mapuoffset := as_err_ok;
end;

// --- read_ ----------------------------------------------------------------

function read_mapvoffset( var f: file ): longint;
var
  mat: as_pmaterial;
  map: as_pmap;
  v: single;
  res: longint;
begin
  mat := as_pmaterial( c_node );
  blockread( f, v, sizeof( v ), res );
  if ( res <> sizeof( v )) then
    begin
      read_mapvoffset := as_err_invalid_file;
      exit;
    end;
  map := nil;
  case ( c_chunk_last ) of
    chunk_texture:
      map := addr( mat^.texture );
    chunk_bumpmap:
      map := addr( mat^.bump );
    chunk_reflection:
      map := addr( mat^.reflection );
    end;
  if ( map <> nil ) then
    map^.vofs := v;
  read_mapvoffset := as_err_ok;
end;

// --- read_ ----------------------------------------------------------------

function read_maprotangle( var f: file ): longint;
var
  mat: as_pmaterial;
  map: as_pmap;
  a: single;
  res: longint;
begin
  mat := as_pmaterial( c_node );
  blockread( f, a, sizeof( a ), res );
  if ( res <> sizeof( a )) then
    begin
      read_maprotangle := as_err_invalid_file;
      exit;
    end;
  map := nil;
  case ( c_chunk_last ) of
    chunk_texture:
      map := addr( mat^.texture );
    chunk_bumpmap:
      map := addr( mat^.bump );
    chunk_reflection:
      map := addr( mat^.reflection );
    end;
  if ( map <> nil ) then
    map^.angle := a;
  read_maprotangle := as_err_ok;
end;

// === keyframer ============================================================

function read_frames( var f: file ): longint;
var
  c: array[ 0..1 ] of longint;
  res: longint;
begin
  blockread( f, c, sizeof( c ), res );
  if ( res <> sizeof( c )) then
    begin
      read_frames := as_err_invalid_file;
      exit;
    end;
  as3d_scene^.framestart := c[0];
  as3d_scene^.frameend := c[1];
  read_frames := as_err_ok;
end;

function read_objnumber( var f: file ): longint;
var
  n: word;
  res: longint;
begin
  blockread( f, n, sizeof( n ), res );
  if ( res <> sizeof( n )) then
    begin
      read_objnumber := as_err_invalid_file;
      exit;
    end;
  c_id := n;
  read_objnumber := as_err_ok;
end;

function read_dummyname( var f: file ): longint;
var
  obj: as_pobject;
begin
  obj := as_pobject( c_node );
  if ( read_asciiz( f ) <> as_err_ok ) then
    begin
      read_dummyname := as_err_invalid_file;
      exit;
    end;
  obj^.fname := c_string;
  read_dummyname := as_err_ok;
end;

function read_trackobjname( var f: file ): longint;
var
  node: as_pwnode;
  pnode: as_pknode;
  obj: as_pobject;
  tobj: as_pobjecttrack;
  cam: as_pcamera;
  tcam: as_pcameratrack;
  tctg: as_pcamtargettrack;
  lit: as_plight;
  tomni: as_pomnilighttrack;
  tspot: as_pspotlighttrack;
  tstrg: as_pspottargettrack;
  amb: as_pambient;
  tamb: as_pambienttrack;
  track: pointer;
  flags: array[ 0..1 ] of word;
  parent: integer;
  wparent, res: longint;
begin
  wparent := -1;
  if ( c_chunk_prev <> chunk_objnumber ) then
    inc( c_id );

  if ( read_asciiz( f ) <> as_err_ok ) then
    begin
      read_trackobjname := as_err_invalid_file;
      exit;
    end;

  if ( c_string = '$AMBIENT$' ) then
    begin
      new( amb );
      if ( amb = nil ) then
        begin
          read_trackobjname := as_err_out_of_memory;
          exit;
        end;
      amb^.fname := c_string;
      amb^.id := 1024 + c_id;
      amb^.color.r := 0.0;
      amb^.color.g := 0.0;
      amb^.color.b := 0.0;
      as3d_addworld( as_obj_ambient, amb );
    end else
  if ( c_string = '$$$DUMMY' ) then
    begin
      new( obj );
      if ( obj = nil ) then
        begin
          read_trackobjname := as_err_out_of_memory;
          exit;
        end;
      obj^.id := 1024 + c_id;
      obj^.flags := as_obj_dummy;
      obj^.numverts := 0;
      obj^.numfaces := 0;
      obj^.verts := nil;
      obj^.faces := nil;
      as_vect_zero( obj^.translate );
      as_vect_zero( obj^.scale );
      as_quat_zero( obj^.rotate );
      as3d_addworld( as_obj_object, obj );
    end else begin
      as3d_byname( node, c_string );
      if ( node = nil ) then
        begin
          read_trackobjname := as_err_invalid_object;
          exit;
        end;
      obj := as_pobject( node^.obj );
      cam := as_pcamera( node^.obj );
      lit := as_plight( node^.obj );
    end;
  blockread( f, flags, sizeof( flags ), res );
  if ( res <> sizeof( flags )) then
    begin
      read_trackobjname := as_err_invalid_file;
      exit;
    end;
  blockread( f, parent, sizeof( parent ), res );
  if ( res <> sizeof( parent )) then
    begin
      read_trackobjname := as_err_invalid_file;
      exit;
    end;
  if ( parent <> -1 ) then
    begin
      pnode := as3d_scene^.keyfr;
      while ( pnode <> nil ) do
        begin
          if ( pnode^.id = parent ) then
            wparent := as_pobject( pnode^.obj )^.id;
          pnode := pnode^.next;
        end;
    end;
  case ( c_chunk_last ) of
    chunk_trackinfo:
      begin
        obj^.parent := wparent;
        if (( flags[0] and $0800 ) > 0 ) then
          obj^.flags := obj^.flags or as_obj_chidden;
        new( tobj );
        if ( tobj = nil ) then
          begin
            read_trackobjname := as_err_out_of_memory;
            exit;
          end;
        fillchar( tobj^, sizeof( as_objecttrack ), 0 );
        as3d_addtrack( as_trk_object, c_id, parent, tobj, obj );
        c_node := obj;
      end;
    chunk_trackcamera:
      begin
        cam^.parent1 := wparent;
        new( tcam );
        if ( tcam = nil ) then
          begin
            read_trackobjname := as_err_out_of_memory;
            exit;
          end;
        fillchar( tcam^, sizeof( as_cameratrack ), 0 );
        as3d_addtrack( as_trk_camera, c_id, parent, tcam, cam );
      end;
    chunk_trackcamtgt:
      begin
        cam^.parent2 := wparent;
        new( tctg );
        if ( tctg = nil ) then
          begin
            read_trackobjname := as_err_out_of_memory;
            exit;
          end;
        fillchar( tctg^, sizeof( as_camtargettrack ), 0 );
        as3d_addtrack( as_trk_camtarget, c_id, parent, tctg, cam );
      end;
    chunk_tracklight:
      begin
        lit^.parent1 := wparent;
        new( tomni );
        if ( tomni = nil ) then
          begin
            read_trackobjname := as_err_out_of_memory;
            exit;
          end;
        fillchar( tomni^, sizeof( as_omnilighttrack ), 0 );
        as3d_addtrack( as_trk_omnilight, c_id, parent, tomni, lit );
      end;
    chunk_trackspotl:
      begin
        lit^.parent1 := wparent;
        new( tspot );
        if ( tspot = nil ) then
          begin
            read_trackobjname := as_err_out_of_memory;
            exit;
          end;
        fillchar( tspot^, sizeof( as_spotlighttrack ), 0 );
        as3d_addtrack( as_trk_spotlight, c_id, parent, tspot, lit );
      end;
    chunk_trackligtgt:
      begin
        lit^.parent2 := wparent;
        new( tstrg );
        if ( tstrg = nil ) then
          begin
            read_trackobjname := as_err_out_of_memory;
            exit;
          end;
        fillchar( tstrg^, sizeof( as_spottargettrack ), 0 );
        as3d_addtrack( as_trk_spottarget, c_id, parent, tstrg, lit );
      end;
    chunk_ambientkey:
      begin
        new( tamb );
        if ( tamb = nil ) then
          begin
            read_trackobjname := as_err_out_of_memory;
            exit;
          end;
        fillchar( tamb^, sizeof( as_ambienttrack ), 0 );
        as3d_addtrack( as_trk_ambient, c_id, parent, tamb, amb );
      end;
    end;
  read_trackobjname := as_err_ok;
end;

function read_trackpivot( var f: file ): longint;
var
  obj: as_pobject;
  pos: array[ 0..2 ] of single;
  i, res: longint;
begin
  obj := as_pobject( c_node );
  blockread( f, pos, sizeof( pos ), res );
  if ( res <> sizeof( pos )) then
    begin
      read_trackpivot := as_err_invalid_file;
      exit;
    end;
{}
  as_vect_make( obj^.pivot, pos[0], pos[1], pos[2] );
  as_vect_swap( obj^.pivot );
  for i := 0 to obj^.numverts-1 do
    as_vect_sub( obj^.verts[i].vert, obj^.verts[i].vert, obj^.pivot );
{}
  read_trackpivot := as_err_ok;
end;

function read_kflags( var f: file; var nf: word; key: as_pkey ): longint;
var
  unknown, flags: word;
  i, res: longint;
  dat: single;
begin
  key^.tens := 0.0;
  key^.cont := 0.0;
  key^.bias := 0.0;
  key^.easeto := 0.0;
  key^.easefrom := 0.0;
  blockread( f, nf, sizeof( word ), res );
  if ( res <> sizeof( word )) then
    begin
      read_kflags := as_err_invalid_file;
      exit;
    end;
  blockread( f, unknown, sizeof( word ), res );
  if ( res <> sizeof( word )) then
    begin
      read_kflags := as_err_invalid_file;
      exit;
    end;
  blockread( f, flags, sizeof( word ), res );
  if ( res <> sizeof( word )) then
    begin
      read_kflags := as_err_invalid_file;
      exit;
    end;
  for i := 0 to 15 do
    begin
      if (( flags and ( 1 shl i )) > 0 ) then
        begin
          blockread( f, dat, sizeof( dat ), res );
          if ( res <> sizeof( dat )) then
            begin
              read_kflags := as_err_invalid_file;
              exit;
            end;
          case ( i ) of
            0: key^.tens := dat;
            1: key^.cont := dat;
            2: key^.bias := dat;
            3: key^.easeto := dat;
            4: key^.easefrom := dat;
            end;
        end;
    end;
  read_kflags := as_err_ok;
end;

function read_tflags( var f: file; track: as_ptrack; var n: word ): longint;
var
  flags: array[ 0..6 ] of word;
  res: longint;
begin
  blockread( f, flags, sizeof( flags ), res );
  if ( res <> sizeof( flags )) then
    begin
      read_tflags := as_err_invalid_file;
      exit;
    end;
  if (( flags[0] and $02 ) = $02 ) then
    track^.flags := as_trk_repeat;
  if (( flags[0] and $03 ) = $03 ) then
    track^.flags := as_trk_loop;
  n := flags[5];
  read_tflags := as_err_ok;
end;

function read_trackpos( var f: file ): longint;
var
  track: as_ptrack;
  key: as_pkey;
  pos: array[ 0..2 ] of single;
  n, nf: word;
  i, res: longint;
begin
  track := alloc_track;
  if ( read_tflags( f, track, n ) <> as_err_ok ) then
    begin
      read_trackpos := as_err_invalid_file;
      exit;
    end;
  for i := 0 to n-1 do
    begin
      new( key );
      if ( key = nil ) then
        begin
          read_trackpos := as_err_out_of_memory;
          exit;
        end;
      if ( read_kflags( f, nf, key ) <> as_err_ok ) then
        begin
          read_trackpos := as_err_invalid_file;
          exit;
        end;
      blockread( f, pos, sizeof( pos ), res );
      if ( res <> sizeof( pos )) then
        begin
          read_trackpos := as_err_invalid_file;
          exit;
        end;
      key^.val.x := pos[0];
      key^.val.y := pos[1];
      key^.val.z := pos[2];
      as_quat_swap( key^.val );
      add_key( track, key, nf );
    end;
  as_spline_init( track );
  as3d_settrack( as_key_position, c_id, track );
  read_trackpos := as_err_ok;
end;

function read_trackcolor( var f: file ): longint;
var
  track: as_ptrack;
  key: as_pkey;
  pos: array[ 0..2 ] of single;
  n, nf: word;
  i, res: longint;
begin
  track := alloc_track;
  if ( read_tflags( f, track, n ) <> as_err_ok ) then
    begin
      read_trackcolor := as_err_invalid_file;
      exit;
    end;
  for i := 0 to n-1 do
    begin
      new( key );
      if ( key = nil ) then
        begin
          read_trackcolor := as_err_out_of_memory;
          exit;
        end;
      if ( read_kflags( f, nf, key ) <> as_err_ok ) then
        begin
          read_trackcolor := as_err_invalid_file;
          exit;
        end;
      blockread( f, pos, sizeof( pos ), res );
      if ( res <> sizeof( pos )) then
        begin
          read_trackcolor := as_err_invalid_file;
          exit;
        end;
      key^.val.x := pos[0];
      key^.val.y := pos[1];
      key^.val.z := pos[2];
      // [BUG] Swap RGB? Needed?!...
      as_quat_swap( key^.val );
      add_key( track, key, nf );
    end;
  as_spline_init( track );
  as3d_settrack( as_key_color, c_id, track );
  read_trackcolor := as_err_ok;
end;

function read_trackrot( var f: file ): longint;
var
  track: as_ptrack;
  key: as_pkey;
  q, old: as_quatern;
  pos: array[ 0..3 ] of single;
  keys, n, nf: word;
  i, res: longint;
  angle: single;
begin
  track := alloc_track;
  as_quat_identity( old );
  if ( read_tflags( f, track, n ) <> as_err_ok ) then
    begin
      read_trackrot := as_err_invalid_file;
      exit;
    end;
  keys := n;
  for i := 0 to n-1 do
    begin
      new( key );
      if ( key = nil ) then
        begin
          read_trackrot := as_err_out_of_memory;
          exit;
        end;
      if ( read_kflags( f, nf, key ) <> as_err_ok ) then
        begin
          read_trackrot := as_err_invalid_file;
          exit;
        end;
      blockread( f, pos, sizeof( pos ), res );
      if ( res <> sizeof( pos )) then
        begin
          read_trackrot := as_err_invalid_file;
          exit;
        end;
      as_quat_fromang( q, pos[0], pos[1], pos[2], pos[3] );
      as_quat_swap( q );
      // [FIX] Angle is absolute!...
      if ( i = 0 ) then
        angle := pos[0] else
        angle += pos[0];
      as_quat_make( key^.val, angle, pos[1], pos[2], pos[3] );
      as_quat_swap( key^.val );
      as_quat_mul( old, q, old );
      as_quat_copy( key^.qa, old );
      add_key( track, key, nf );
    end;
  as_spline_initrot( track );
  as3d_settrack( as_key_rotate, c_id, track );
  read_trackrot := as_err_ok;
end;

function read_trackscale( var f: file ): longint;
var
  track: as_ptrack;
  key: as_pkey;
  pos: array[ 0..2 ] of single;
  n, nf: word;
  i, res: longint;
begin
  track := alloc_track;
  if ( read_tflags( f, track, n ) <> as_err_ok ) then
    begin
      read_trackscale := as_err_invalid_file;
      exit;
    end;
  for i := 0 to n-1 do
    begin
      new( key );
      if ( key = nil ) then
        begin
          read_trackscale := as_err_out_of_memory;
          exit;
        end;
      if ( read_kflags( f, nf, key ) <> as_err_ok ) then
        begin
          read_trackscale := as_err_invalid_file;
          exit;
        end;
      blockread( f, pos, sizeof( pos ), res );
      if ( res <> sizeof( pos )) then
        begin
          read_trackscale := as_err_invalid_file;
          exit;
        end;
      key^.val.x := pos[0];
      key^.val.y := pos[1];
      key^.val.z := pos[2];
      as_quat_swap( key^.val );
      add_key( track, key, nf );
    end;
  as_spline_init( track );
  as3d_settrack( as_key_scale, c_id, track );
  read_trackscale := as_err_ok;
end;

function read_trackfov( var f: file ): longint;
var
  track: as_ptrack;
  key: as_pkey;
  fov: single;
  n, nf: word;
  i, res: longint;
begin
  track := alloc_track;
  if ( read_tflags( f, track, n ) <> as_err_ok ) then
    begin
      read_trackfov := as_err_invalid_file;
      exit;
    end;
  for i := 0 to n-1 do
    begin
      new( key );
      if ( key = nil ) then
        begin
          read_trackfov := as_err_out_of_memory;
          exit;
        end;
      if ( read_kflags( f, nf, key ) <> as_err_ok ) then
        begin
          read_trackfov := as_err_invalid_file;
          exit;
        end;
      blockread( f, fov, sizeof( fov ), res );
      if ( res <> sizeof( fov )) then
        begin
          read_trackfov := as_err_invalid_file;
          exit;
        end;
      key^.val.w := fov;
      add_key( track, key, nf );
    end;
  as_spline_init( track );
  as3d_settrack( as_key_fov, c_id, track );
  read_trackfov := as_err_ok;
end;

function read_trackroll( var f: file ): longint;
var
  track: as_ptrack;
  key: as_pkey;
  roll: single;
  n, nf: word;
  i, res: longint;
begin
  track := alloc_track;
  if ( read_tflags( f, track, n ) <> as_err_ok ) then
    begin
      read_trackroll := as_err_invalid_file;
      exit;
    end;
  for i := 0 to n-1 do
    begin
      new( key );
      if ( key = nil ) then
        begin
          read_trackroll := as_err_out_of_memory;
          exit;
        end;
      if ( read_kflags( f, nf, key ) <> as_err_ok ) then
        begin
          read_trackroll := as_err_invalid_file;
          exit;
        end;
      blockread( f, roll, sizeof( roll ), res );
      if ( res <> sizeof( roll )) then
        begin
          read_trackroll := as_err_invalid_file;
          exit;
        end;
      key^.val.w := roll;
      add_key( track, key, nf );
    end;
  as_spline_init( track );
  as3d_settrack( as_key_roll, c_id, track );
  read_trackroll := as_err_ok;
end;

function read_trackmorph( var f: file ): longint;
var
  track: as_ptrack;
  key: as_pkey;
  node: as_pwnode;
  n, nf: word;
  i, res: longint;
begin
  node := nil;
  track := alloc_track;
  if ( read_tflags( f, track, n ) <> as_err_ok ) then
    begin
      read_trackmorph := as_err_invalid_file;
      exit;
    end;
  for i := 0 to n-1 do
    begin
      new( key );
      if ( key = nil ) then
        begin
          read_trackmorph := as_err_out_of_memory;
          exit;
        end;
      if ( read_kflags( f, nf, key ) <> as_err_ok ) then
        begin
          read_trackmorph := as_err_invalid_file;
          exit;
        end;
      if ( read_asciiz( f ) <> as_err_ok ) then
        begin
          read_trackmorph := as_err_invalid_file;
          exit;
        end;
      as3d_byname( node, c_string );
      if ( node = nil ) then
        begin
          read_trackmorph := as_err_invalid_object;
          exit;
        end;
      key^.val.w := as_pobject( node^.obj )^.id;
      add_key( track, key, nf );
    end;
  as3d_settrack( as_key_morph, c_id, track );
  read_trackmorph := as_err_ok;
end;

function read_trackhide( var f: file ): longint;
var
  track: as_ptrack;
  key: as_pkey;
  unknown: array[ 0..1 ] of word;
  n, nf: word;
  hide, i, res: longint;
begin
  hide := 0;
  track := alloc_track;
  if ( read_tflags( f, track, n ) <> as_err_ok ) then
    begin
      read_trackhide := as_err_invalid_file;
      exit;
    end;
  for i := 0 to n-1 do
    begin
      new( key );
      if ( key = nil ) then
        begin
          read_trackhide := as_err_out_of_memory;
          exit;
        end;
      if ( read_kflags( f, nf, key ) <> as_err_ok ) then
        begin
          read_trackhide := as_err_invalid_file;
          exit;
        end;
      blockread( f, unknown, sizeof( unknown ), res );
      if ( res <> sizeof( unknown )) then
        begin
          read_trackhide := as_err_invalid_file;
          exit;
        end;
      hide := 1-hide;
      key^.val.w := hide;
      add_key( track, key, nf );
    end;
  as3d_settrack( as_key_hide, c_id, track );
  read_trackhide := as_err_ok;
end;

function read_chunk( var f: file; h: pchunk ): longint;
var
  res: longint;
begin
  blockread( f, h^.id, sizeof( word ), res );
  if ( res <> sizeof( word )) then
    begin
      read_chunk := as_err_invalid_file;
      exit;
    end;
  blockread( f, h^.size, sizeof( longint ), res );
  if ( res <> sizeof( longint )) then
    begin
      read_chunk := as_err_invalid_file;
      exit;
    end;
  read_chunk := as_err_ok;
end;

function chunkreaderworld( var f: file; p: longint; parent: word ): longint;
var
  h: chunk;
  pc: longint;
  n, i, err: integer;
begin
  c_chunk_last := parent;
  pc := filepos( f );
  while ( pc < p ) do
    begin
      if ( read_chunk( f, @h ) <> as_err_ok ) then
        begin
          chunkreaderworld := as_err_invalid_file;
          exit;
        end;
      c_chunk_curr := h.id;
      n := -1;
      for i := 0 to 50 do
        if ( h.id = chunk_world[i].id ) then
          n := i;
      if ( n < 0 ) then
        begin
          pc := pc + h.size;
          seek( f, pc );
        end else begin
          pc += h.size;
{$ifdef as_3ds_debug}
  assign( debug, 'debug3ds' );
  append( debug );
  writeln( debug, dbstr, 'chunk_world[', n, '].id = ', hex( h.id, 4 ));
  close( debug );
{$endif}
          err := chunk_world[n].func( f );
          if ( err <> as_err_ok ) then
            begin
              chunkreaderworld := err;
              exit;
            end;
          if ( chunk_world[n].sub > 0 ) then
            begin
{$ifdef as_3ds_debug}
  dbstr := dbstr + '  ';
{$endif}
              err := chunkreaderworld( f, pc, h.id );
{$ifdef as_3ds_debug}
  dbstr := copy( dbstr, 1, length( dbstr )-2 );
{$endif}
              if ( err <> as_err_ok ) then
                begin
                  chunkreaderworld := err;
                  exit;
                end;
            end;
          seek( f, pc );
          c_chunk_prev := h.id;
        end;

      if ( pc <> filepos( f )) then
        begin
          chunkreaderworld := as_err_invalid_file;
          exit;
        end;

    end;
  chunkreaderworld := as_err_ok;
end;

function chunkreaderkey( var f: file; p: longint; parent: word ): longint;
var
  h: chunk;
  pc: longint;
  n, i, err: integer;
begin
  c_chunk_last := parent;
  pc := filepos( f );
  while ( pc < p ) do
    begin
      if ( read_chunk( f, @h ) <> as_err_ok ) then
        begin
          chunkreaderkey := as_err_invalid_file;
          exit;
        end;
      c_chunk_curr := h.id;
      n := -1;
      for i := 0 to 21 do
        if ( h.id = chunk_keyfr[i].id ) then
          n := i;
      if ( n < 0 ) then
        begin
          pc += h.size;
          seek( f, pc );
        end else begin
          pc += h.size;
{$ifdef as_3ds_debug}
  assign( debug, 'debug3ds' );
  append( debug );
  writeln( debug, dbstr, 'chunk_keyfr[', n, '].id = ', hex( h.id, 4 ));
  close( debug );
{$endif}
          err := chunk_keyfr[n].func( f );
          if ( err <> as_err_ok ) then
            begin
              chunkreaderkey := err;
              exit;
            end;
          if ( chunk_keyfr[n].sub > 0 ) then
            begin
{$ifdef as_3ds_debug}
  dbstr := dbstr + '  ';
{$endif}
              err := chunkreaderkey( f, pc, h.id );
{$ifdef as_3ds_debug}
  dbstr := copy( dbstr, 1, length( dbstr )-2 );
{$endif}
              if ( err <> as_err_ok ) then
                begin
                  chunkreaderkey := err;
                  exit;
                end;
            end;
          seek( f, pc );
          c_chunk_prev := h.id;
        end;
      if ( pc <> filepos( f )) then
        begin
          chunkreaderkey := as_err_invalid_file;
          exit;
        end;
    end;
  chunkreaderkey := as_err_ok;
end;

function as3d_loadmesh3ds( var f: file ): longint;
var
  version: byte;
  len, res: longint;
begin
{$ifdef as_3ds_debug}
  assign( debug, 'debug3ds' );
  append( debug );
  writeln( debug, dbstr, 'as_3ds.loadmesh3ds()...' );
  close( debug );
  dbstr := dbstr + '  ';
{$endif}
  c_id := 0;
  len := filesize( f );
  seek( f, 28 );
  blockread( f, version, 1, res );
  if ( res <> 1 ) then
    begin
      as3d_loadmesh3ds := as_err_invalid_file;
      exit;
    end;
  if ( version < 2 ) then
    begin
      as3d_loadmesh3ds := as_err_invalid_version;
      exit;
    end;
  seek( f, 0 );
  res := chunkreaderworld( f, len, 0 );
{$ifdef as_3ds_debug}
  dbstr := copy( dbstr, 1, length( dbstr )-2 );
  assign( debug, 'debug3ds' );
  append( debug );
  writeln( debug, dbstr, 'as_3ds.loadmesh3ds(): ', as3d_geterror( res ));
  close( debug );
{$endif}
  as3d_loadmesh3ds := res;
end;

function as3d_loadmotion3ds( var f: file ): longint;
var
  version: byte;
  len, res: longint;
begin
{$ifdef as_3ds_debug}
  assign( debug, 'debug3ds' );
  append( debug );
  writeln( debug, dbstr, 'as_3ds.loadmotion3ds()...' );
  close( debug );
  dbstr := dbstr + '  ';
{$endif}
  c_id := -1;
  len := filesize( f );
  seek( f, 28 );
  blockread( f, version, 1, res );
  if ( res <> 1 ) then
    begin
      as3d_loadmotion3ds := as_err_invalid_file;
      exit;
    end;
  if ( version < 2 ) then
    begin
      as3d_loadmotion3ds := as_err_invalid_version;
      exit;
    end;
  seek( f, 0 );
  res := chunkreaderkey( f, len, 0 );
{$ifdef as_3ds_debug}
  dbstr := copy( dbstr, 1, length( dbstr )-2 );
  assign( debug, 'debug3ds' );
  append( debug );
  writeln( debug, dbstr, 'as_3ds.loadmotion3ds(): ', as3d_geterror( res ));
  close( debug );
{$endif}
  as3d_loadmotion3ds := res;
end;

end.